'use strict';

/**
 * @ngdoc service
 * @name modules.arcgis.services:ArcGisJob
 * @description
 *     Gestion des job ArcGIS. ces jobs sont des services de géotraitement ESRI.
 *     Une fois le job lancé, on le suit pour connaître son état et récupérer les messages.
 *     Les messages sont préparés pour être affichés dans la fenêtre des détails du job.
 *
 */
define(function() {
  const ArcGisJob = function($q, $filter, gaDomUtils, $timeout, ArcGisGeoprocess) {

    /**
     * Ajoute, à la liste présentée dans la fenêtre des détails sur
     * l'exécution du job, les nouveaux messages du compte rendu
     * de l'état d'exécution du job ESRI.
     */
    const manageMessages = (scope,jsonRes) => {
      if (!jsonRes.messages || !scope.jobDetailMessages) {
        return;
      }
      for (let ind = scope.jobDetailMessages.length;
        ind < jsonRes.messages.length; ind++) {
        scope.jobDetailMessages.push(jsonRes.messages[ind]);
      }
    };


    /**
     * Mise en place dans le scope d'un status pour la fabrication
     * de l'ensemble des atlas.
     * On a un arcgisJobs[jobId] par atlas.
     *
     * @param {*} scope Scope qui contient déjà les infos de chaque atlas, et
     *                   dans lequel on inscrit un statut global
     */
    const defineGlobalStatus = (scope) => {
      if (scope.arcgisJobs) {
        let failedCount = 0;
        let warnCount = 0;
        let jobCount = 0;
        // -- Parcours des jobs arcgis (de fabrication des atlas)
        // -- pour compter le nombre qui ont échoué ou levé des alertes
        for (const prop in scope.arcgisJobs) {
          jobCount++;
          if (scope.arcgisJobs[prop].jobInfoStatus) {
            if (scope.arcgisJobs[prop].jobInfoStatus==='failed') {
              failedCount++;
            }
            if (scope.arcgisJobs[prop].jobInfoStatus.includes('.warning')) {
              warnCount++;
            }
          }
        }
        // -- Définir un statut global fonction des alertes ou échecs trouvés
        scope.jobsStatus = scope.jobInfoStatus;
        if (jobCount!==0) {
          if (failedCount!==0) {
            if (scope.jobInfoStatus === 'failed' && failedCount===jobCount) {
              scope.jobsStatus = 'failed';
            }
            else {
              scope.jobsStatus = 'warning';
            }
          }
          else if (warnCount!==0 && scope.jobInfoStatus!=='failed') {
            scope.jobsStatus = 'warning';
          }
        }
      }
    };


    /**
     * Définition de l'état du job en cours d'exécution en fonction
     * soit de l'état du job lui même, soit de l'état d'un des messages.
     */
    const setJobStatus = (scope, jsonRes) => {
      let jobDesc;
      if (scope.arcgisJobs && scope.arcgisJobs[jsonRes.jobId]) {
        jobDesc = scope.arcgisJobs[jsonRes.jobId];
      }
      else {
        jobDesc = scope;
      }
      const jobStatus = jsonRes.jobStatus;
      manageMessages(jobDesc,jsonRes);
      if (jobStatus === 'esriJobFailed') {
        jobDesc.jobInfoStatus = 'failed';
        jobDesc.executingJob = false;
      }
      else {
        if (jobStatus === 'esriJobSucceeded') {
          jobDesc.jobInfoStatus = 'succeeded';
          jobDesc.executingJob = false;
        }
        else if (jobStatus=== 'esriJobExecuting') {
          jobDesc.jobInfoStatus = 'executing';
          jobDesc.executingJob = true;
        }
        // -- Quel que soit l'un des 2 cas ci-dessus, si un message
        // -- d'alerte est présent on considére l'état du service en alerte.
        if (jobDesc.jobDetailMessages && jobDesc.jobDetailMessages.find(
          message => message.type === 'esriJobMessageTypeWarning')) {
          jobDesc.jobInfoStatus += '.warning';
        }
      }
      jobDesc.readableJobInfoStatus
          =  $filter('translate')('arcgisJob.' + jobStatus)
          .replace('$1',scope.arcgisJobName);
      defineGlobalStatus(scope);
    };



    /**
     * Affichage d'un SWeet ALert message pour indiquer que le job d'atalas a échoué.
     *
     * @param {*} scope Scope du widget à l'origine du (ou des) jobs.
     * @param {*} mess Message à afficher
     * @param {*} params Informations reçues depuis l'action initiatrice du job
     */
    const esriAtlasFailureMessage = (scope, messDetails, params) => {
      if (!scope.noJobMessage) {
        gaDomUtils.hideGlobalLoader();
        let mess = $filter('translate')('arcgisJob.failureOfAction') || '';

        // -- KIS-3460
        // -- -- Il faudrait afficher un message générique reprenant le nom de l’action
        // -- -- => Titre : “Erreur d’exécution“   Détails : “Echec de l’action ‘Export Inspection’”
        if (params && params.actionName) {
          mess = mess.replace('$2',':').replace('$1',params.actionName);
        }
        else {
          mess = mess.replace('$2','').replace('$1','');
        }
        if (messDetails) {
          mess += ' -> ' + messDetails;
        }

        swal({
          title: $filter('translate')('arcgisJob.jobFailed'),
          text: mess,
          type: 'error',
          closeOnConfirm: false,
          closeOnCancel: false
        });
      }
    };


    /**
     * Initialise le scope avec les information du job que l'on vient de lancer.
     * Si un job est déjà en cours, on continue de travailler avec, mais
     * il est déplacé dans scope.arcgisJobs. Dans cet objet une propriété est
     * la description du déroulement du job. Le nom de la propriété
     * qui accéde à la descripition du job est son id (le job id).
     *
     * @param {*} scope Scope du widget appelant
     */
    const initJob = (scope, jsonRes) => {

      // -- Au moins un 2 eme job lancé en simultané pour ce scope
      if (!scope.arcgisJobs) {
        scope.arcgisJobs = {};
      }
      scope.arcgisJobs[jsonRes.jobId] = {
        jobDetailMessages: [],
        jobInfo: jsonRes
      };
    };


    const storeResult = (scope, jobId, paramName, value) => {
      if (!scope.arcgisJobs[jobId].jobResults) {
        scope.arcgisJobs[jobId].jobResults = [];
      }
      scope.arcgisJobs[jobId].jobResults.push({name: paramName, value: value});
    };


    const downloadResultFile = (scope, serviceUrl, jsonRes, outFileParamNames,
      ind, paramType, params, defer) => {
      if (!defer) {
        defer = $q.defer();
      }
      if (!outFileParamNames || !outFileParamNames.length || ind>=outFileParamNames.length) {
        defer.resolve(null);
      }
      else { // if (ind<outFileParamNames.length)
        // -- Job Arcgis terminé avec succés
        let fileUrl = serviceUrl + '/jobs/' + jsonRes.jobId + '/';
        const paramName = outFileParamNames[ind];
        if (jsonRes.results[paramName]) {
          fileUrl += jsonRes.results[outFileParamNames[ind]].paramUrl + '?f=pjson';
          const promise = $.get(fileUrl);
          promise.then((newres) => {
            const jsonRes2 = JSON.parse(newres);
            if (paramType==='GPDataFile') {
              storeResult(scope, jsonRes.jobId, paramName+' (URL)', jsonRes2.value.url);
              // -- Sous entendu que l'appelant ne fait qu'un seul open,
              // -- celui sur le premier fichier à télécharger
              if (params.actionForFile) {
                if (params.actionForFile==='window.open') {
                  window.open(jsonRes2.value.url);
                }
                else {
                  let param;
                  if (params.actionParams) {
                    param = {
                      url: jsonRes2.value.url,
                      params: params.actionParams
                    };
                  }
                  else {
                    param = jsonRes2.value;
                  }
                  params.actionForFile(param);
                }
              }
            }
            else {
              storeResult(scope, jsonRes.jobId, jsonRes2.paramName, jsonRes2.value);
            }
            downloadResultFile(scope, serviceUrl, jsonRes, outFileParamNames, ind+1,
              paramType, params, defer);
          });
        }
      }
      return defer.promise;
    };


    /**
     * Récupérer depuis ArcGIS les résultatus du géotraitement décrit par jsonres.
     * Le résultat est mis à disposition dans le scope (scope.arcgisJob[jobId]).
     *
     * @param {type} scope - description du scope
     * @param {type} serviceUrl - description de l'URL du service de géotraitement
     * @param {type} jsonRes - description de la réponse JSON
     * @param {type} params - description des paramètres
     * @return {type} une promesse qui se résout lorsque les résultats du travail sont récupérés
     */
    const getJobResults = (scope, serviceUrl, jsonRes, params) => {
      const defer = $q.defer();
      ArcGisGeoprocess.getOutFile(serviceUrl).then((outFileParamNames) => {
        downloadResultFile(scope, serviceUrl, jsonRes, outFileParamNames,
          0, 'GPDataFile', params).then(() => {
          ArcGisGeoprocess.getSimpleOutParams(serviceUrl).then((simpleOutParams) => {
            if (simpleOutParams) {
              downloadResultFile(scope, serviceUrl, jsonRes, simpleOutParams,
                0, 'GPString', params).then(() => {
                defer.resolve();
              });
            }
            else {
              defer.resolve();
            }
          });
        });
      });
      return defer.promise;
    };


    /**
     * Lance un job ESRI qui est un géotraitement.
     * Relance une consultation toute les seconde
     * pour connaître l'état d'avancement du job.
     *
     *
     * @param {*} scope  Scope du widget ou autre objet pour récupérer les informations
     *                  de déroulement du job et/ou pour demander à ne pas avoir
     *                  de swal qui s'affiche (scope.noJobMessage=true)
     * @param {*} serviceUrl URL du géotraitement ArcGIS
     * @param {*} res Réponse à la requête HTTP
     * @param {*} outParamName  Nom du paramétre en sortie
     *                          du géotraitement à exploiter
     * @param {*} defer Defer qui redonnera la main à celui qui a demandé
     *              l'exécution du géotraitement une fois celui-ci terminé.
     * @returns La promesse qui attend la fin du géotraitement pour
     *          éventuellement retourner le résultat
     */
    const runEsriJob = (scope, serviceUrl, res, params, defer) => {

      let jsonRes;
      if (typeof res == 'string')
        jsonRes = JSON.parse(res);
      else
        jsonRes = res;

      if (!defer) {
        defer = $q.defer();
        initJob(scope, jsonRes);
      }

      setJobStatus(scope, jsonRes);
      if (jsonRes.jobStatus == 'esriJobFailed') {
        // -- Echec d'exécution du job ArcGIS
        esriAtlasFailureMessage(scope, '', params);
        defer.resolve(jsonRes);
      }
      else if (jsonRes.error) {
        // -- Erreur à l'exécution du job ARcGIS
        esriAtlasFailureMessage(scope, jsonRes.error.message, params);
        defer.resolve(jsonRes);
      }
      else if (jsonRes.results) {
        if (jsonRes.jobStatus == 'esriJobSucceeded') {
          getJobResults(scope, serviceUrl, jsonRes, params).then(() => {
            defer.resolve(jsonRes);
          });
        } else {
          // -- Job ArcGIS terminé avec erreur
          esriAtlasFailureMessage(scope, '', params);
          defer.resolve(jsonRes);
        }
      }
      else {
        // -- Job non terminé: lancement d'interrogation sur l'état de celui-ci dans une seconde
        $timeout(() => {
          let url = serviceUrl + '/jobs' + '/' + jsonRes.jobId + '?f=pjson';
          $.get(url).then((newres) => {
            runEsriJob(scope, serviceUrl, newres, params, defer);
          });
        }, 1000);
      }
      return defer.promise;
    };


    return {
      runEsriJob: runEsriJob
    };
  };

  ArcGisJob.$inject = ['$q', '$filter', 'gaDomUtils', '$timeout', 'ArcGisGeoprocess' ];
  return ArcGisJob;
});
