'use strict';
define(() => {
  const importexportwidget = function (
    FeatureTypeFactory,
    FeatureAttachmentFactory,
    ImportExportFactory,
    $filter,
    $timeout,
    $q,
    SelectManager,
    $interval
  ) {
    return {
      templateUrl:
        'js/XG/widgets/mapapp/importdocuments/views/importdocumentswidget.html',
      restrict: 'A',

      link: scope => {
        const allowedFeatureTypesInfo = ['POLYGON', 'LINE', 'POINT'];
        const allowedFeatureAttributeTypes = ['g2c.attachments', 'g2c.attachment'];
        const allowedFileTypes = ['.zip'];

        const loadingTypes = {
          upload: 'upload',
          download: 'download'
        }

        init();

        scope.setZipFile = event => {
          if (!isValidFile(event.files[0])) { return false; }
          try {
            scope.$apply(() => {
              scope.data.file = event.files[0];
            });
          } catch (err) {
            delete scope.data.file;
            showError('importDocumentsWidget.file.brokenFile');
          }
          event.value = ''; // Empty the input so the user can re-upload the file;
        }

        scope.downloadAttributeAttachments = () => {
          if (!canAddProcess(scope.data, loadingTypes.download)) {
            showError('importDocumentsWidget.download.existing');
            return;
          }
          const loadingProcess = addProcess(scope.data, loadingTypes.download);
          downloadFileAttachment(loadingProcess);

        }

        scope.removeAttributeAttachments = () => {
          const confirmRemove = window.confirm(
            translate('importDocumentsWidget.attributes.removeWarning')
          );
          if (!confirmRemove) { return; }

          let selectedLayersOfFeature = getSelectedFeatureLayers();
          let confirmRemoveSelected = false;
          if (selectedLayersOfFeature.length) {
            confirmRemoveSelected = window.confirm(
              translate('importDocumentsWidget.attributes.removeSelectedWarning')
            );
          }
          if (!confirmRemoveSelected) {
            selectedLayersOfFeature = undefined;
          }

          FeatureAttachmentFactory.deleteFeaturesAttachments(
            scope.data.feature.name, 
            scope.data.attribute.name, 
            selectedLayersOfFeature
          ).then(() => {
              require('toastr').success(
                translate('importDocumentsWidget.attributes.removedMessage')
              );
          }).catch(() => showError('importDocumentsWidget.attributes.removedErrorMessage'));
        }

        /**
         * Soumet le fichier pour import et mapping des pièces jointes
         * Vérifie d'abord si un processus est déjà en cours
         * Ajoute un nouveau processus si possible
         * Supprime le fichier du scope après soumission
         */
        scope.submit = () => {
          if (!scope.data.file) {
            showError('importDocumentsWidget.file.errors.missing');
            return;
          }
          if (!canAddProcess(scope.data, loadingTypes.upload)) {
            showError('importDocumentsWidget.upload.errors.existing');
            return;
          }
          const loadingProcess = addProcess(scope.data, loadingTypes.upload);
          uploadFileAndMapAttachments(loadingProcess);
          
          // Supprimer directement la référence au fichier sans utiliser $apply
          delete scope.data.file;
          
          // Réinitialisation du champ file sans déclencher de validation
          document.getElementById('importDocZipFile').value = '';
        }

        scope.retryProcess = (loadingProcess) => {
          switch (loadingProcess.type) {
            case loadingTypes.upload:
              return uploadFileAndAttach(loadingProcess);
            case loadingTypes.download:
              return downloadFileAttachment(loadingProcess);
          }
          throw 'retryProcess type not found!!!';
        }

        function init() {
          scope.features = [];
          scope.attributes = [];
          scope.data = {};
          scope.loadProcesses = [];
          scope.acceptedFileTypes = allowedFileTypes.join(', ');
          scope.loadingTypes = loadingTypes;

          resolveFeatures();
        }

        function canAddProcess(data, type) {
          switch (type) {
            case loadingTypes.upload:
              return !scope.loadProcesses.filter(process => process.type === type &&
                process.file.name === data.file.name &&
                process.feature.name === data.feature.name &&
                process.attribute.name === data.attribute.name &&
                !process.finished
              ).length;
            case loadingTypes.download:
              return !scope.loadProcesses.filter(process => process.type === type &&
                process.feature.name === data.feature.name &&
                process.attribute.name === data.attribute.name &&
                !process.finished
              ).length;
          }
          return false;
        }

        function addProcess(data, type) {
          scope.loadProcesses.unshift(Object.assign({ type }, data));
          return scope.loadProcesses[0];
        }

        function resolveFeatures() {
          FeatureTypeFactory.get()
            .then(res => {
              // adapte aux modification de FeatureFactory.get (doit utiliser res et non plus res.data)
              if (!res || !Array.isArray(res)) throw {};
              scope.features = res.filter(feature => (
                allowedFeatureTypesInfo.indexOf(feature.typeInfo) !== -1 &&
                getFeatureDocumentsAttributes(feature).length > 0
              ));
              scope.$watch('data.feature', featureChanged);
            })
            .catch(() => showError('importDocumentsWidget.features.errors.failGet'))
            .finally(() => scope.loadedFeatures = true);
        }

        function uploadFileAndAttach(data) {
          data.error = false;
          data.finished = false;
          FeatureAttachmentFactory
            .importFeatureAttachmentsArchive(data.file, data.feature.name, data.attribute.name)
            .then((res) => FeatureAttachmentFactory
              .mapFeaturesAttachments(data.file.name, scope.data.feature.name, scope.data.attribute.name, scope.data.removeOrphans)
            )
            .then(res => {
              data.foundFiles = res.data.receivedFilesCount;
              data.modifiedFiles = res.data.mappedAttachmentsCount;
              data.removedOrphans = res.data.removedOrphanReferencesCount;
              data.journalUrl = getDownloadAttachmentZipUrl(res.data.mappingStatusFile);
              delete scope.data.file;
            })
            .catch((err) => {
              showError('importDocumentsWidget.upload.errors.fail');
              data.error = true;
              throw err;
            })
            .finally(() => data.finished = true);
        }

        function downloadFileAttachment(data) {
          data.error = false;
          data.finished = false;
          FeatureAttachmentFactory.downloadFeaturesAttachments(data.feature.name, data.attribute.name)
            .then((res) => {
              if (!res || !res.data || !res.data.processId) { throw false; }
              return checkDownloadProcess(res.data.processId)
                .then(attachRes => {
                  const downloadUrl = getDownloadAttachmentZipUrl(attachRes.data.archiveId);
                  window.open(downloadUrl);
                  data.downloadUrl = downloadUrl;
                });
            })
            .catch(err => {
              data.error = true;
              showError('importDocumentsWidget.download.errors.fail');
              throw err;
            }).finally(() => data.finished = true);
        }

        function checkDownloadProcess(processId) {
          const retryTimeout = 3000;
          return FeatureAttachmentFactory.checkDownloadAttributeStatus(processId)
            .then(res => {
              if (!res || !res.data) { throw false; }
              switch (res.data.attachmentsReady) {
                case 'false':
                  const defer = $q.defer();
                  $timeout(() => defer.resolve(checkDownloadProcess(processId)), retryTimeout);
                  return defer.promise;
                case 'true':
                case 'N/A':
                  return res;
              }
              throw false; // In case the server doesn't return 'true'/'false'/'N/A' throw error
            });
        }

        function getDownloadAttachmentZipUrl(attachmentId) {
          return `/services/${localStorage.portal}/export/downloadexportedfile?f=json&exportedFileId=${attachmentId}`;
        }

        function featureChanged() {
          scope.data.attribute = undefined;
          scope.attributes = getFeatureDocumentsAttributes(scope.data.feature);
        }

        function getFeatureDocumentsAttributes(feature = { attributes: [] }) {
          return feature.attributes.filter(
            attribute => allowedFeatureAttributeTypes.indexOf(attribute.type) !== -1
          );
        }

        // returns false if file is not zip and throws error for user
        function isValidFile(file) {
          if (
            !file.name ||
            !allowedFileTypes.filter(allowedFileType => file.name.endsWith(allowedFileType)).length
          ) {
            showError('importDocumentsWidget.file.errors.invalidFormat', {allowedFormats: scope.acceptedFileTypes});
            return false;
          }
          return true;
        }

        function getSelectedFeatureLayers() {
          const selectedFeatures = SelectManager.getFeatures();
          const searchedFeatures = selectedFeatures.features
            ? selectedFeatures.features.filter(feature => feature.id.split('.')[0])
            : [];
          return searchedFeatures.map(feature => feature.id);
        }

        function showError(translationKey, translationOptions) {
          require('toastr').error(translate(translationKey, translationOptions));
        }

        function translate(translationKey, options) {
          return $filter('translate')(translationKey, options);
        }

        /**
         * Gère l'upload du fichier et le mapping des pièces jointes
         * @param {Object} data - Les données du processus d'upload
         */
        const uploadFileAndMapAttachments = (data) => {
          data.error = false;
          data.finished = false;
          data.progress = 2;
          data.message = 'Réception de l\'archive...';
          
          FeatureAttachmentFactory.importAndMapAttachments(
            data.file,
            data.feature.name,
            data.attribute.name,
            data.removeOrphans
          ).then(processUid => {
            // Démarrage du suivi de progression
            checkProcessStatus(processUid, data);
          }).catch(() => {
            data.error = true;
            showError('importDocumentsWidget.upload.errors.fail');
            data.finished = true;
          });
        }

        /**
         * Vérifie périodiquement l'état du processus d'import
         * @param {string} processUid - L'UID du processus à vérifier
         * @param {Object} data - Les données du processus
         */
        const checkProcessStatus = (processUid, data) => {
          // Création d'un interval pour vérifier l'état du processus
          const interval = $interval(() => {
            FeatureAttachmentFactory.checkImportProcessStatus(processUid)
              .then(res => {
                const process = res.data;
                if (process.progress < 100 && process.etat.toLowerCase() === 'running') {
                  // Mise à jour de la progression
                  data.progress = process.progress;
                  data.message = process.messageToProcess;
                } else {
                  // Traitement terminé, on arrête l'interval
                  $interval.cancel(interval);
                  data.finished = true;
                  if (process.errorMessage) {
                    data.error = true;
                    showError(process.errorMessage);
                  } else {
                    const result = JSON.parse(process.messageToProcess);
                    data.foundFiles = result.receivedFilesCount;
                    data.modifiedFiles = result.mappedFiles ? result.mappedFiles.length : 0;
                    data.removedOrphans = result.orphansRemoved ? result.orphansRemoved.length : 0;
                    data.journalUrl = getDownloadAttachmentZipUrl(result.mappingStatusFile);
                  }
                }
              })
              .catch((error) => {
                console.error(error);
                // En cas d'erreur, on arrête l'interval
                $interval.cancel(interval);
                data.error = true;
                showError('importDocumentsWidget.upload.errors.fail');
                data.finished = true;
              });
          }, 1000); // Vérification toutes les secondes

          // Nettoyage de l'interval lors de la destruction du scope
          scope.$on('$destroy', () => {
            $interval.cancel(interval);
          });
        }
      },
    };
  };

  importexportwidget.$inject = [
    'FeatureTypeFactory',
    'FeatureAttachmentFactory',
    'ImportExportFactory',
    '$filter',
    '$timeout',
    '$q',
    'SelectManager',
    '$interval'
  ];
  return importexportwidget;
});
