/**@module layerManager*/
'use strict';
define(function () {
  const mapModelSelector = ($rootScope, $filter, ParametersFactory, gaJsUtils, layersService) => {
    return {
      templateUrl: 'js/XG/widgets/mapapp/layerManager/mapmodel/views/mapModelSelector.html',
      scope: {
        map: '=',
        spinner: '='
      },
      restrict: 'EA',
      link: function (scope) {

        /**
         * Exécute la méthode de layersService pour appliquer à la carte le mapModel fourni en paramètre.
         * L'id du mapModel est enregistré dans le service en tant que modèle actif
         */
        scope.onMapModelChange = () => {
          if (scope.mapModel) {

            // affiche le spinner du sélecteur de modèles de carte
            scope.spinner.show = true;
            layersService.setMapModel(scope.mapModel);

          } else if (Array.isArray(scope.mapModels) && scope.mapModels.length > 0) {

            // Après ajout/update/suppr d'un modèle de carte depuis le manager, le tableau est mis à jour,
            // ce qui provoque l'exécution de onMapModelChange avec mapModel === null.
            // Il faut alors afficher le nom du modèle actif dans le select.
            setSelectedModel();
          }
        };

        /**
         * Récupère la liste des modèles de carte.
         * Les modèles de carte n'ont jamais fait l'objet d'une attention particulière:<ul><li>
         * pas de factory</li><li>
         * pas de tableau stocké dans la factory</li><li>
         * une dizaine d'appels API similaires augmentant la latence de l'initialisation de KIS</li></ul>
         * Re-écrire le fichier gcMainController n'est pas l'objet du ticket KIS-2844
         * @param {boolean} forceGetMapModels true pour récupérer la liste des modèles depuis le serveur.
         * Par défaut, le paramètre est égal à true. Il est false dans le cas d'un clic sur le bouton "Réinitialiser le modèle de carte" (@see scope.restoreMapModel)
         */
        scope.getMapModels = (forceGetMapModels = true) => {
          const srid = scope.map.getView().getProjection().getCode();
          if (forceGetMapModels) {
            ParametersFactory.getbytype(layersService.MAP_MODELS_USERS, srid).then(
                res => {
                  scope.mapModels = res.data;
                  setSelectedModel();
                },
                err => {
                  require('toastr').error(
                      $filter('translate')('applications.applications.mapModelServerError'));
                  console.error('getMapModels : ', err.data);
                }
            );
          } else if (Array.isArray(scope.mapModels)) {
            setSelectedModel();
          } else {
            scope.getMapModels(true);
          }
        };

        /**
         * Défini le modèle de carte affiché dans le select à partir de l'id du modèle actif présent dans le service
         */
        const setSelectedModel = () => {
          if (Number.isInteger(layersService.resources.activeModel)) {
            const index = scope.mapModels.findIndex(param => param.data.id === layersService.resources.activeModel);
            if (index > -1) {
              scope.mapModel = scope.mapModels[index];
            }
          }
        };


        /**
         * Au clic sur le bouton réinitialiser.
         * Lance l'exécution de la méthode gcMainController.setDefaultMap
         */
        scope.restoreMapModel = () => {
          scope.mapModel = null;
          // à la restauration, on n'a pas besoin de rechercher les modèles de carte dans le serveur
          scope.$emit('setDefaultMap', false);
        };


        /**
         * A l'ajout/suppression d'un modèle de carte dans mapModelMgtWidget.
         * La mise à jour de la liste des modèles de carte se fait:
         *   - par évènement car on fait avec ce que l'on a dans AngularJS
         *   - en utilisant $rootScope car KIS est fait à l'arrache. Obligé de souiller cette directive à peine créée.
         */
        $rootScope.$on('mapModelListUpdated', scope.getMapModels);


        /**
         * Au clic sur le bouton "Activer le modèle" dans le manager des modèles de carte (mapModelMgtWidget),
         * la méthode layersService.setMapModel envoie un évènement "gcUpdateMainConfig" au gcMainController qui envoie un évènement "mapModelChange"
         * mapmodelmgtwidget => layersService => gcMainController => mapModelSelector  : sélectionne le modèle actif</li></ul>
         */
        scope.$on('mapModelChange', setSelectedModel);

        /******************
         * INITIALISATION *
         ******************/

        /**
         * Quand gcMainController a défini le modèle de carte (aucun ou celui par défaut)
         */
        scope.$on('mapModelReady', (evt, forceGetMapModels) => {
          scope.getMapModels(forceGetMapModels);
        });
      },
    };
  };

  mapModelSelector.$inject = ['$rootScope', '$filter', 'ParametersFactory', 'gaJsUtils',
    'layersService'];
  return mapModelSelector;
});
