/**
 * Objet JSAppiaConnect à utiliser pour communiquer avec les produits APPIA
 *
 * @param _oOperisConnect Objet Flex Appia-Connect à utiliser pour la communication
 *
 */
function JSAppiaConnect(_oOperisConnect) {
  //================== Attributs

  /**
   * Objet Flex OperisConnect
   */
  this.__oOperisConnect = _oOperisConnect;

  /**
   * Version d'Appia-Connect
   */
  this.__version = 'undefined yet';
}

//================== Constantes

/**
 * Canal AppiaConnect
 */
JSAppiaConnect.prototype.CANAL_APPIACONNECT = '_AppiaConnect';

//------------------ Applications

/**
 * Identifiant d'application VisDGI
 */
JSAppiaConnect.prototype.APPLICATION_VISDGI = '_VisDGI';

/**
 * Identifiant d'application DDC
 */
JSAppiaConnect.prototype.APPLICATION_DDC = '_DroitsdeCites';

/**
 * Identifiant d'application Oxalis
 */
JSAppiaConnect.prototype.APPLICATION_OXALIS = '_Oxalis';

/**
 * Identifiant d'application UrbaPro
 */
JSAppiaConnect.prototype.APPLICATION_URBAPRO = '_UrbaPro';

//------------------ Identifiants de fonctions

/**
 * Fonction DDC RechercherDossiersParcelles
 */
JSAppiaConnect.prototype.FONCTION_DDC_RECHERCHER_DOSSIERS_PARCELLES =
  'ddcConnectRechercherDossiersParcelles';

/**
 * Fonction OperisConnect RechercherDossiersParcelles
 */
JSAppiaConnect.prototype.FONCTION_RECHERCHER_DOSSIERS_PARCELLES =
  'opeConnectRechercherDossiersParcelles';

/**
 * Fonction DDC RetourImprimerDossier
 */
JSAppiaConnect.prototype.FONCTION_DDC_RETOUR_IMPRIMER_DOSSIER =
  'ddcConnectRetourImpressionDossier';

/**
 * Fonction DDC AfficherFicheParcelle
 */
JSAppiaConnect.prototype.FONCTION_DDC_AFFICHER_FICHE_PARCELLE =
  'ddcConnectAfficherFicheParcelle';

/**
 * Fonction DDC AfficherFicheDossier
 */
JSAppiaConnect.prototype.FONCTION_DDC_AFFICHER_FICHE_DOSSIER =
  'ddcConnectAfficherFicheDossier';

/**
 * Fonction VISDGI RechercherParcelles
 */
JSAppiaConnect.prototype.FONCTION_VISDGI_RECHERCHER_PARCELLES =
  'ddcConnectRechercherParcelles';

/**
 * Fonction VISDGI RechercherParcelles
 */
JSAppiaConnect.prototype.FONCTION_VISDGI_AFFICHER_RELEVE_PROPRIETE =
  'ddcConnectAfficherRelevePropriete';

/**
 * Fonction PLU MisAJourReglementParcelle
 */
//appel limité à 40K, donc on signal le début et la fin des lots de parcelle/règlements par un paramètre
JSAppiaConnect.prototype.FONCTION_PLU_MAJ_REGLEMENT_PARCELLE =
  'ddcConnectRetourMajReglementsParcelle';

/**
 * Fonction PLU IMC MisAJourReglementParcelle
 */
//appel limité à 40K, donc on signal le début et la fin des lots de parcelle/règlements par un paramètre
JSAppiaConnect.prototype.FONCTION_PLU_IMC_ARCOPOLE =
  'ddcConnectOperis_RetourIMC';

//------------------ Erreurs

/**
 * Préfixe des messages d'erreur
 */
JSAppiaConnect.prototype.ERREUR_PREFIXE = 'Erreur : ';

/**
 * Message d'erreur lorsque l'objet Appia-Connect n'est pas défini
 */
JSAppiaConnect.prototype.MESSAGE_ERREUR_APPIA_CONNECT_INCONNU =
  JSAppiaConnect.prototype.ERREUR_PREFIXE +
  "L'objet Appia-Connect n'est pas défini";

/**
 * Code d'erreur lorsque l'objet Appia-Connect n'est pas défini
 */
JSAppiaConnect.prototype.CODE_ERREUR_APPIA_CONNECT_INCONNU = -1000;

//================== Fonctions DDC (JS -> DDC)

/**
 * Vérifications avant le lancement d'une fonction
 *
 * @param _application Identifiant de l'application
 * @param _message     Potentiel message d'erreur
 *
 */
JSAppiaConnect.prototype.preVerifications = function(_application, _message) {
  // Vérification de la présence de l'objet Appia Connect
  if (this.__oOperisConnect == null) {
    _message = JSAppiaConnect.MESSAGE_ERREUR_APPIA_CONNECT_INCONNU;
    return JSAppiaConnect.CODE_ERREUR_APPIA_CONNECT_INCONNU;
  }

  _message = null;
  return 0;
};

/**
 * Appel Initialisation : InitCanaux
 *
 *
 */
JSAppiaConnect.prototype.InitCanaux = function() {
  this.__oOperisConnect.InitCanaux(this.APPLICATION_DDC);
};

/**
 * Appel Initialisation : InitCanal
 *
 * @param _canal l'application à avertir que le SIG est prêt
 */
JSAppiaConnect.prototype.InitCanal = function() {
  this.__oOperisConnect.InitCanal(this.CANAL_APPIACONNECT);
};

/**
 * Appel DDC : RechercherDossiersParcelles
 *
 * @param _dossiers  Tableau des dossiers à rechercher
 * @param _parcelles Tableau des parcelles à rechercher
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.DDC_RechercherDossiersParcelles = function(
  _dossiers,
  _parcelles,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_DDC_RECHERCHER_DOSSIERS_PARCELLES,
    [_dossiers, _parcelles]
  );
};

/**
 * Appel Operis : RechercherDossiersParcelles
 * @param canal le canal à utiliser
 * @param _dossiers  Tableau des dossiers à rechercher
 * @param _parcelles Tableau des parcelles à rechercher
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.OP_RechercherDossiersParcelles = function(
  canal,
  _dossiers,
  _parcelles,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(canal, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    canal,
    this.FONCTION_RECHERCHER_DOSSIERS_PARCELLES,
    [_dossiers, _parcelles]
  );
};

/**
 * Appel DDC : RetourImprimerDossier
 *
 * @param _url URL de l'image retournée
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.DDC_RetourImprimerDossier = function(
  _dossier,
  _parcelles,
  _url,
  _idImage,
  _echelle,
  _largeur,
  _hauteur,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_DDC_RETOUR_IMPRIMER_DOSSIER,
    [_dossier, _parcelles, _url, _idImage, _echelle, _largeur, _hauteur]
  );
};

/**
 * Appel DDC : AfficherFicheParcelle
 *
 * @param _parcelle Parcelle dont on demande la fiche
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.DDC_AfficherFicheParcelle = function(
  _parcelle,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_DDC_AFFICHER_FICHE_PARCELLE,
    [_parcelle]
  );
};

/**
 * Appel DDC : AfficherFicheDossier
 *
 * @param _dossier Dossier dont on demande la fiche
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.DDC_AfficherFicheDossier = function(
  _dossier,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;
  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_DDC_AFFICHER_FICHE_DOSSIER,
    [_dossier]
  );
};

/**
 * Appel VISDGI : RechercherParcelles
 *
 * @param _parcelles Tableau des parcelles à rechercher
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.VISDGI_RechercherParcelles = function(
  _parcelles,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_VISDGI, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_VISDGI,
    this.FONCTION_VISDGI_RECHERCHER_PARCELLES,
    [_parcelles]
  );
};

/**
 * Appel VISDGI : RelevePropriete
 *
 * @param _parcelle Parcelle dont on demande le relevé de propriété
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.VISDGI_AfficherRelevePropriete = function(
  _parcelle,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_VISDGI, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_VISDGI,
    this.FONCTION_VISDGI_AFFICHER_RELEVE_PROPRIETE,
    [_parcelle]
  );
};

/**
 * Appel DdC : MajReglements du dossier en cours, pour chaque parcelle
 *
 * @param _parcelle Parcelle dont on demande le relevé de propriété
 * @param _reglements tableau de 1 ou plus règlement, à l'appelant de vérifier si la taille ne dépasse pas 40K en tout
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur, ou PLU_MajReglementsParcelle_debut ou PLU_MajReglementsParcelle_fin
 * pour indiquer le début ou la fin
 */
JSAppiaConnect.prototype.PLU_MajReglementsParcelle = function(
  _parcelle,
  _reglements,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_PLU_MAJ_REGLEMENT_PARCELLE,
    [_parcelle, _reglements, _codeErreur, _message]
  );
};

/**
 * Appel DdC : IMC Arcopole du dossier en cours
 *
 * @param _dossier Dossier concerné
 * @param _xml XML retourné par Arcopole
 * @param _urlxml url de l'XML retourné par Arcopole
 * @param _codeErreur Code d'erreur (0 si OK)
 * @param _message Potentiel message d'erreur
 */
JSAppiaConnect.prototype.Operis_RetourIMC = function(
  _xml,
  _urlxml,
  _dossier,
  _codeErreur,
  _message
) {
  // Vérifications préalables
  _codeErreur = this.preVerifications(this.APPLICATION_DDC, _message);
  if (_codeErreur < 0) return;

  this.__oOperisConnect.AppelFonction(
    this.APPLICATION_DDC,
    this.FONCTION_PLU_IMC_ARCOPOLE,
    [_dossier, _xml, _urlxml, _codeErreur, _message]
  );
};
