'use strict';
define(() => {
  /**
   * Directive du fil d'ariane de la partie Edition d'un formulaire de type Intervention Simple
   * @param gaJsUtils pour vérifier la non nullité de variables
   * @param isUtils contient la majorité de la logique du fil d'ariane
   * @param FeatureTypeFactory pour récupérer un fti par son nom
   * @param $filter pour traduire du texte
   * @param $rootScope permet d'aller chercher la source de données principale du portail
   * @param $q permet de grouper des appels api
   * @return {{scope: {res: string, isMode: string, curtemplate: string}, link: link, restrict: string, templateUrl: string}}
   */
  let isBreadcrumb = (gaJsUtils, isUtils, FeatureTypeFactory, $filter, $rootScope, $q, $timeout) => {
    return {
      templateUrl: 'js/XG/widgets/utilities/form/views/isBreadcrumb.html',
      restrict: 'A',
      scope: {
        curtemplate: '=', // objet data du formulaire IS en cours
        isMode: '=',      // 'render'/'builder'
        res: '=',         // conteneur des variables du formulaire
        ftiuid: '='       // uid du composant maître de l'IS
      },
      link: (scope, elt) => {

        /**
         * Au clic sur le nom d'une étape du fil d'ariane, active l'étape cliquée.
         * Inactif en mode création
         * @param {number} id propriété id de la page cliquée à activer
         */
        scope.changeIsEditionActiveStep = (id) => {
          if (scope.isMode === 'builder' || (scope.isMode === 'render'
              && gaJsUtils.notNullAndDefined(scope.res, 'current.id')
              && ((scope.curtemplate.breadcrumb.currentStep === 0 && isGeometryStepValid())
              || scope.curtemplate.breadcrumb.currentStep > 0))) {

            scope.curtemplate.breadcrumb.currentStep = id;
            scope.curtemplate.breadcrumb.goBack = true;
            scope.curtemplate.breadcrumb.goNext = true;

            const popupContainer = elt[0].closest('.popup.popupContainer');

            if (id === 2) {
              isUtils.addIsListenersAndObservers(popupContainer);

              // ajuste la hauteur du corps de l'onglet en fonction dunombre de lignes d'onglets
              if (elt && elt[0]) {
                $timeout(() => {
                  isUtils.adjustTabContentHeight(popupContainer.id, 3);
                });
              }
            }

            scope.$parent.$parent.$parent.$parent.$broadcast('setInitialDataTableHeight');

            // KIS-3194: afficher la page "Objets sélectionnés" uniquement en présence d'objets sélectionnés
            if (id === 4) {
              createSelectedObjectsList();
            }

            // masque le bouton "étape suivante" au clic sur la dernière étape
            // masque le bouton "étape précédente" au clic sur la 1ère étape
            const nextStepIndex = scope.curtemplate.breadcrumb.steps.findIndex(step => step.id === id);


            if (nextStepIndex > -1) {
              if (nextStepIndex === scope.curtemplate.breadcrumb.steps.length - 1) {
                scope.curtemplate.breadcrumb.goNext = false;
              } else if (nextStepIndex === 0) {
                scope.curtemplate.breadcrumb.goBack = false;
              }
            }
          }
        };

        /**
         * Modifie la valeur de la variable correspondant à l'id de l'étape à activer.
         * Permet de naviger vers l'étape suivante ou l'étape précédente.
         * L'ordre des étapes est différent entre le mode création et le mode modification.
         * @param {object} current objet en cours de création/édition dans l'intervention simple
         * @param {number} currentStep id de l'étape actuellement active
         * @param {string} direction sens dans lequel on souhaite naviguer ("previous"/"forward")
         * @param {HTMLDivElement} popupContainer conteneur principal de la popup
         */
        const isNavigateStep = (current, currentStep, direction, popupContainer) => {

          // récupère l'ordre des étapes du fil d'ariane
          const navigationArray = gaJsUtils.notNullAndDefined(current, 'id') ? isUtils.navigationArray.edition : isUtils.navigationArray.creation;

          // récupère l'id de la prochaine étape dans la direction fournie
          const currentIndex = navigationArray.findIndex(id => id === currentStep);
          if (currentIndex > -1) {
            const nextIndex = typeof direction === 'string' && direction === 'previous' ? currentIndex - 1 : currentIndex + 1;

            if (navigationArray[nextIndex] === 2) {
              isUtils.addIsListenersAndObservers(popupContainer);

              // ajuste la hauteur du corps de l'onglet en fonction dunombre de lignes d'onglets
              if (elt && elt[0]) {
                const iddiv = elt[0].closest('.popup.popupContainer').id;
                $timeout(() => {
                  isUtils.adjustTabContentHeight(iddiv, 3);
                });
              }
            }

            // KIS-3194: afficher la page "Objets sélectionnés" uniquement en présence d'objets sélectionnés
            if (navigationArray[nextIndex] === 4) {
              createSelectedObjectsList();
            }

            return navigationArray[nextIndex];
          } else {
            console.error('isNavigateStep : aucune étape existe ayant l\'id ', currentStep);
            return currentStep;
          }
        };

        /**
         * Au clic sur le bouton "Etape précédente" du fil d'ariane de l'IS (flèche droite),
         * modifie la valeur de l'étape active (currentStep) d'après l'ordre des étapes.
         * L'ordre des étapes est différent entre le mode création et le mode modification.
         */
        scope.previous = () => {

          // Calcule l'id de l'étape suivante à activer
          scope.curtemplate.breadcrumb.currentStep = isNavigateStep(scope.res.current, scope.curtemplate.breadcrumb.currentStep, 'previous');

          // Affiche le bouton "étape suivante" (flèche droite)
          scope.curtemplate.breadcrumb.goNext = true;

          // Masque le bouton étape précédente (flèche gauche) si l'étape active est la 1ère étape du fil d'ariane
          const previousStepIndex = scope.curtemplate.breadcrumb.steps.findIndex(step => step.id === scope.curtemplate.breadcrumb.currentStep);

          if (previousStepIndex === 0) {
            scope.curtemplate.breadcrumb.goBack = false;
          }
        };

        /**
         * Au clic sur le bouton "Etape suivante" du fil d'ariane de l'IS (flèche droite),<ul><li>
         * assure la validation de l'étape courante</li><li>
         * modifie la valeur de l'étape active (currentStep) d'après l'ordre des étapes</li></ul>
         * L'ordre des étapes est différent entre le mode création et le mode modification.
         *
         */
        scope.forward = () => {
          if ((scope.curtemplate.breadcrumb.currentStep === 0 && isGeometryStepValid()) || scope.curtemplate.breadcrumb.currentStep > 0) {

            const popupContainer = elt[0].closest('.popupContainer');

            // Calcule l'id de l'étape suivante à activer
            scope.curtemplate.breadcrumb.currentStep = isNavigateStep(scope.res.current, scope.curtemplate.breadcrumb.currentStep, 'forward', popupContainer);

            // Affiche le bouton "étape précédente" (flèche gauche)
            scope.curtemplate.breadcrumb.goBack = true;

            const nextStepIndex = scope.curtemplate.breadcrumb.steps.findIndex(step => step.id === scope.curtemplate.breadcrumb.currentStep);

            // Affiche le nom de l'étape suivante dans le fil d'ariane
            scope.curtemplate.breadcrumb.steps[nextStepIndex].show = true;
            scope.moreStepsShown = isUtils.hasSeveralStepsShown(scope.curtemplate.breadcrumb.steps);

            // A la dernière étape, on masque le bouton "étape suivante" (flèche droite)
            if (nextStepIndex > -1 && nextStepIndex === scope.curtemplate.breadcrumb.steps.length - 1) {
              scope.curtemplate.breadcrumb.goNext = false;
            }
          }
        };

        const showError = (titleKey, textKey) => { 
          swal({
            text: $filter('translate')(textKey),
            title: $filter('translate')(titleKey),
            type: 'error',
            showCancelButton: false,
          });
        };
        /**
         * Validation de l'étape "Géométrie" lors de la création d'un objet dans un formulaire IS.<ul><li>
         * Si l’option « Géometrie obligatoire pour valider l'enregistrement » a été activée (configuration dans le bouton Enregistrer)
         * alors le bouton « Suivant » bloque l’accès à l’étape suivante et affiche une popup d’information « Vous devez sélectionner
         * les objets du patrimoine pour poursuivre. »</li><li>
         * Si l’option n’est pas activée ou que la sélection a été faite alors le bouton amène au volet Objets sélectionnés.</li></ul>
         */
        const isGeometryStepValid = () => {
          if (!scope.res.current.id) {
            const mustHaveGeometry = scope.curtemplate.savefield.config.mustHaveGeometry;
            const autoCreateGeometry = scope.curtemplate.savefield.config.autoCreateGeometry;
            if (mustHaveGeometry && !autoCreateGeometry && (!scope.res.current.geometry || scope.res.current.geometry === null)) {
              //Si l'option "Géométrie obligatoire pour valider l'enregistrement" est activée
              showError('tools.builder.intervention_simple.edition.mustHaveGeometry', 'tools.builder.intervention_simple.edition.UseProvidedTools');
              return false;
            }
            if( autoCreateGeometry && !mustHaveGeometry && !isUtils.isHasSelectedFeatures(scope.curtemplate, scope.res)
            ) {
              //Si l'option "Créer la géométrie à partir des objets en association" est activée
              showError('tools.builder.intervention_simple.edition.autoCreateGeometry', 'tools.builder.intervention_simple.edition.UseProvidedTools');
              return false;
            } 
            if(autoCreateGeometry && mustHaveGeometry && !isUtils.isHasSelectedFeatures(scope.curtemplate, scope.res) && (!scope.res.current.geometry || scope.res.current.geometry === null)){
              // Si les deux cases sont cochées, on vérifie que l'une au moins existe sinon on affiche un message d'erreur
              showError('tools.builder.intervention_simple.edition.mustHaveGeometryOrAutoCreate', 'tools.builder.intervention_simple.edition.UseProvidedTools');
              return false;
            }
          }
          return true;
        };

        /**
         * Créé une propriété de la variable res pour stocker le tableau des objets sélectionnés
         * qui sera affiché dans la datatable de l'étape "Objets sélectionnés"
         */
        const createSelectedObjectsList = () => {

          // prépare une nouvelle propriété de l'objet res pour stocker le tableau des objets sélectionnés à présenter dans l'étape "Objets sélectionnés"
          scope.res.isSelectedObj = {features: [], totalFeatures: 0};

          if (!scope.res.selectedObjectsFakeFti || Object.entries(scope.res.selectedObjectsFakeFti).length === 0) {
            // prépare les colonnes de la table des objets sélectionnés
            scope.res.selectedObjectsFakeFti =  isUtils.selectedObjectsFti;
          }

          if (isUtils.hasSelectedFeatures) {

            // récupère les objets sélectionnés à partir des objets en association ou des objets reliés
            let selectedFeatures;
            if (isUtils.hasRelatedOrAssocFeatures(scope.curtemplate, scope.res, 'association')) {
              selectedFeatures = scope.res[scope.curtemplate.savefield.config.association].features;
            } else if (isUtils.hasRelatedOrAssocFeatures(scope.curtemplate, scope.res, 'related')) {
              selectedFeatures = scope.res[scope.curtemplate.savefield.config.related].features;
            }
            if (selectedFeatures) {

              const mainDb = $rootScope.xgos.portal.parameters.mainDB;
              const promises = [];

              // créé un objet simplifié par objet sélectionné
              for (const selectedFeature of selectedFeatures) {
                const selectedFtiName = selectedFeature.id.split('.')[0];
                const selectedFti = FeatureTypeFactory.getFeatureByName(selectedFtiName);
                if (selectedFti) {

                  // le champ Identifiant correspond au champ "AFIELD" ou "BFIELD"
                  // de l'association entre le composant maître (scope.ftiuid) et le composant de l'objet sélectionné (selectedFti)
                  const promise = isUtils.getSelectedObjectIdentifiant(mainDb, scope.ftiuid, selectedFti, selectedFeature);
                  promises.push(promise);
                }
              }
              $q.all(promises).then(
                  res => {
                    for (let i = 0; i < res.length; i++) {

                      if (res[i]) {
                        // affiche uniquement les objets dont on a trouvé l'identifiant (ceux qui font l'objet d'une association patrimoine)
                        // chaque objet simplifié contient des attributs "Identifiant" et "Couche"
                        // l'ordre des résultats est égal à l'ordre des promesses et à l'ordre des objets sélectionnés
                        const lightFeature = {
                          geometry: selectedFeatures[i].geometry,
                          id: selectedFeatures[i].id,
                          properties: {
                            Identifiant: res[i],
                            Couche: selectedFeatures[i].id.split('.')[0],
                            Alias: ''
                          },
                          type: 'Feature'
                        };
                        scope.res.isSelectedObj.features.push(lightFeature);
                        scope.res.isSelectedObj.totalFeatures++;
                      }
                    }
                  }
              );
            }
          }
        };

        /**
         * Lors de la reprogrammation d'une IS, après récupération des objets cibles de l'IS source,
         * les objets cibles de l'IS source deviennent les objets sélectionnés de l'IS créée
         * @see formRender#reprogramObjetsCibles
         */
        scope.$on('reprogrammingWithTargetFeatures', createSelectedObjectsList);

        /**
         * Dans le cas d’une nouvelle sélection d’objets à associer à l’intervention<ul><li>
         * en mode création/reprogrammation, l'étape "Objets sélectionnés" est insérée après l'étape "Géometrie"</li><li>
         * en mode édition, l'étape est insérée avant l'étape "Objets cibles"</li></ul>
         * @param {boolean} isNew est true en mode création/reprogrammation quand current n'a pas d'id
         */
        const insertSelectedObjStep = (isNew) => {
          if (!scope.curtemplate.breadcrumb.steps.find(step => step.id === 4)) {
            let insertIndex = -1;
            if (isNew) {
              insertIndex = scope.curtemplate.breadcrumb.steps.findIndex(step => step.id === 0);
            } else {
              insertIndex = scope.curtemplate.breadcrumb.steps.findIndex(step => step.id === 1);
            }

            if (insertIndex > -1) {

              // en mode création, c'est après le rang de l'étape géométrie que doit être insérée l'étape "Objets sélectionnés"
              insertIndex = isNew ? insertIndex + 1 : insertIndex;

              // insère l'étape "Objets sélectionnés" à l'index souhaité
              isUtils.addSelectedObjStep(scope.curtemplate, insertIndex, !isNew);

              const navigationArray = isNew ? isUtils.navigationArray.creation : isUtils.navigationArray.edition;

              // met à jour le tableau sur lequel repose la navigation previous/forward (<</>>)
              if (!navigationArray.includes(4)) {
                navigationArray.splice(insertIndex, 0, 4);
              }
            }
          }
        };

        /*******************
         * INITIALISATION
         ******************/

        // alignement horizontal du fil d'ariane:
        // 1 ou 2 étape(s) affichée(s): justify-contennt = flex-start
        // plus de 2 étapes affichés: justify-content = space-evenly
        scope.moreStepsShown = isUtils.hasSeveralStepsShown(scope.curtemplate.breadcrumb.steps);

        // à la reprogrammation, la directive est déjà initialisée.
        // formRender doit envoyer un évènement à la reprogrammation pour modifier l'alignement
        scope.$on('initIsBreandcrumbHAlign', () => {
          scope.moreStepsShown = false;
        });

        /**
         * Surveille scope.res pour vérifier si l'IS contient des objets sélectionnés
         */
        scope.$watch('res', () => {
          if (scope.isMode === 'render') {
            isUtils.hasSelectedFeatures = isUtils.isHasSelectedFeatures(scope.curtemplate, scope.res);

            if (isUtils.hasSelectedFeatures) {

              // Dans le cas d’une nouvelle sélection d’objets à associer à l’intervention,
              // le volet Objets sélectionnés est inséré avant l'étape Objets cibles (édition) ou après l'étape Géométrie (création)
              insertSelectedObjStep(!gaJsUtils.notNullAndDefined(scope.res, 'current.id'));
            }
          }
        }, true);
      },
    };
  };

  isBreadcrumb.$inject = ['gaJsUtils', 'isUtils', 'FeatureTypeFactory', '$filter', '$rootScope', '$q', '$timeout'];
  return isBreadcrumb;
});
