/**
 *
 */
'use strict';
define(function() {
  const formFunctionUtils = function() {
    this.$get = function(FeatureTypeFactory, gaJsUtils) {

      /**
       *
       * @param {string} dataKey
       * @return {boolean}
       */
      const hasTabIndex = dataKey => {
        return dataKey.indexOf('[') !== -1 && dataKey.indexOf(']') !== -1;
      };

      /**
       *
       * @param {string} dataKey
       * @return {*}
       */
      const getTabIndex = dataKey => {
        return dataKey.replace(/(^.*\[|\].*$)/g, '');
      };

      /**
       *
       * @param {string} dataKey
       * @return {*}
       */
      const getTabHolder = dataKey => {
        return dataKey.replace(/\[.*?\]\s?/g, '');
      };

      /**
       *
       * @param {string} data
       * @param {string} key
       * @return {null|*}
       */
      const getDataValueByKey = (data, key) => {
        let dataHolder = data;
        const keys = key.split('.');
        for (const key of keys) {
          if (!dataHolder) return null;

          if (dataHolder && hasTabIndex(key)) {
            const tabIndex = getTabIndex(key);
            const realTabHolder = getTabHolder(key);
            dataHolder = dataHolder[realTabHolder][tabIndex];
          }
          else {
            dataHolder = dataHolder[key];
          }
        }

        return angular.copy(dataHolder);
      }

      /**
       * Objet contenant les informations utiles sur les toutes fonctions nécessitant un fti non vide pour leur exécution
       * dans le cas d'un objet non enregistré (où justement le fti fourni par formFieldRender est vide).<br>
       * A chaque fonction correspond ic un objet dont la clé est le nom de fonction et contenant:
       * - <code>index</code>: le rang de l'argument contenant le nom de l'attribut dont cherche à connaitre le fti (ex. "variable source" est le 2ème argument de la fonction setData)
       * - <code>length</code>: le nombre d'arguments dans la fonction (ex. setData possède 4 argument: destination, source, type_func et décodage restriction)
       * @type {{setCountFromCollection: {length: number, index: number}, setData: {length: number, index: number}, setDataFromCollection: {length: number, index: number}}}
       */
      const requiringFtiFunctions = {
        'setData': {
          index: 1,
          length: 4
        },
        'setDataFromCollection': {
          index: 1,
          length: 5
        },
        'setCountFromCollection': {
          index: 0,
          length: 3
        }
      };

      /**
       * Vérifie parmi toutes les fonctions associées au contrôle de formulaire actuellement traité (bouton de formulaire)
       * si l'une d'entre-elles est une fonction nécessitant un fti
       * @param {[string]} expressions fonctions associées au contrôle de formulaire sur l'évènement "onFinish"/"beforeFinish"
       * @return {boolean} <code>true</code> si le contrôle possède une fonction nécessitant un fti
       * parmi les fonctions attachées à un évènement du contrôle de formulaire
       */
      const someFunctionRequireFti = (expressions) => {
        let someFunctionRequireFti = false;

        if (Array.isArray(expressions) && expressions.length > 0) {
          for (const expression of expressions) {

            // recherche la présence d'une fonction nécessitant un fti parmi les fonctions du contrôle de formulaire
            if (typeof expression === 'string' && expression.length > 0) {

              for (const functionName of Object.keys(requiringFtiFunctions)) {
                someFunctionRequireFti = expression.startsWith(functionName)
                    && (hasRestrictionDecodeEnable(functionName, expression)
                    || hasCalculationOperator(functionName, expression));
                // dès qu'une fonction onFinish possède une fonction nécessitant un fti on renvoie true
                if (someFunctionRequireFti) {
                  return someFunctionRequireFti;
                }
              }
            }
          }
        }
        return someFunctionRequireFti;
      };

      /**
       * Vérifie si la fonction fait partie des fonctions répertoriées dans {@link requiringFtiFunctions}.
       * Dans le cas de setData ou setDataFromCollection on vérifie si l'option du décodage de restriction est activée
       *
       * @param {string} functionName nom de la fonction beforeFinish/onFinish à vérifier
       * @param {string} expression corps de la fonction <code>functionName</code> avec ses arguments
       * sous la forme d'une chaîne de caractères
       * @return {boolean} <code>true</code> si la fonction est incluse dans {@link requiringFtiFunctions}.
       * Sinon on renvoie <code>false</code>. Pour les cas particuliers des fonctions setData et
       * setDataFromCollection, <code>true est renvoyé si l'option de décodage de restriction
       * est activée. <code>false</code> si l'option est désactivée.
       */
      const hasRestrictionDecodeEnable = (functionName, expression) => {
        if (Object.keys(requiringFtiFunctions).includes(functionName)) {
          if (functionName === 'setData' || functionName === 'setDataFromCollection') {
            // isole les arguments de la fonction 'functionName'
            const firstParenthesisIndex = expression.indexOf('(',0);
            const secondParenthesisIndex = expression.indexOf(')',0);
            if (firstParenthesisIndex > -1 && secondParenthesisIndex > -1) {
              const setDataArguments = expression.substring(firstParenthesisIndex + 1, secondParenthesisIndex);
              if (setDataArguments.includes(',')) {

                // évalue le dernier argument correspondant à la case à cocher "Décoder une table de restriction"
                return setDataArguments.slice().split(',').pop() === '\'tab-true\'';
              }
            }
          }
          return true;
        }
        return false;
      };

      /**
       * Vérifie si la fonction setDataCollection possède l'un des opérateurs suivants "sum", "average", "min", "max".
       * Dans ce cas, la fonction nécessite un fti pour son traitement
       * @param functionName nom de la fonction
       * @param expression corps de la fonction
       * @return {boolean} true si la fonction contient un opérateur problématique
       */
      const hasCalculationOperator = (functionName, expression) => {
        // isole les arguments de la fonction 'functionName'
        const firstParenthesisIndex = expression.indexOf('(',0);
        const secondParenthesisIndex = expression.indexOf(')',0);
        if (firstParenthesisIndex > -1 && secondParenthesisIndex > -1) {
          const setDataArguments = expression.substring(firstParenthesisIndex + 1, secondParenthesisIndex);

          // isole l'objet de la fonction
          const kisObjStart = setDataArguments.indexOf('{');
          const kisObjEnd = setDataArguments.indexOf('}');
          if (kisObjStart > -1 && kisObjEnd > -1) {
            const kisObj = setDataArguments.substring(kisObjStart, kisObjEnd + 1).replaceAll(';', ',');
            try {
              const kisObjJson = JSON.parse(kisObj);
              return ['sum', 'average', 'min', 'max'].includes(kisObjJson.operator);
            } catch(e) {
              console.error('hasCalculationOperator : ', e);
              return false;
            }
          }
        }
        return false;
      };

      /**
       * <b>Méthode à utiliser uniquement lorsque le <code>scope.fti</code> de formFieldRender est vide à l'exécution de la méthode <code>formFieldRender.evaluateExpression</code></b>.<br>
       * Pour chaque fonction définie dans l'objet {@link requiringFtiFunctions} on récupère le fti de l'attribut saisi dans l'argument de la fonction
       * @param {string[]} expressions tableau contenant toutes les fonctions onFinish/beforeFinish du contrôle de formulaire actuellement traité.
       * Si la fonction ne fait pas partie des fonctions nécessitant un fti alors le fti de cette fonction restera vide.
       * @param {object[]} allTemplateTabs onglets du formulaire contenant tous les champs du formulaire
       * @return {object[]} tableau contenant le fti de l'objet current pour chaque fonction onFinish/beforeFinish du contrôle de formulaire.
       */
      const findRequiredFtis = (expressions, allTemplateTabs) => {
        const requiredFtis = [];

        // boucle sur les fonctions
        for (let functionIndex = 0; functionIndex < expressions.length; functionIndex++) {
          const expression = expressions[functionIndex];

          // extrait le nom de la fonction
          const openParenthesisIndex = expression.indexOf('(',0);
          const functionName = expression.substring(0,openParenthesisIndex);

          // si la fonction est répertoriée dans requiringFtiFunctions (+ gestion cas setData/setDataFromCollection)
          if (hasRestrictionDecodeEnable(functionName, expression) || hasCalculationOperator(functionName, expression)) {
            const functionFti = findFtiFromFormTabs(functionName, expression, allTemplateTabs);
            requiredFtis.push(functionFti);
          } else {
            // si la fonction ne requiert pas de fti, on renvoie le fti vide de formFieldRender
            requiredFtis.push({});
          }
        }

        // renvoie un tableau où les fti sont ordonnées de la même façon que les fonctions dans le tableau "expressions"
        return requiredFtis;
      };


      /**
       * Récupère le fti de la variable source pour la fonction <code>setData</code>
       * Méthode réservée pour la fonction setData uniquement dans le cas où le fti du scope est vide (cas où l'on modifie un objet non enregistré)
       * @param {string} functionName nom de la fonction à évaluer parmi le tableau de fonction du contrôle de formulaire actuellement traité
       * @param {string} expression corps de la fonction <code>functionName</code> avec ses arguments sous la forme d'une chaîne de caractères
       * @param {[object]} allTemplateTabs onglets du formulaire
       * @return {object|{}} fti de la variable source de la fonction <code>setData</code> ou objet vide
       */
      const findFtiFromFormTabs = (functionName, expression, allTemplateTabs) => {

        // Méthodes internes à findFtiFromFields (extractContainerName, extractVariableName, findFieldInFormTabs et findFieldFti)

        /**
         * Récupère le nom de l'objet contenant l'attribut (attributeName)
         * (ex. on recherche "current" quand l'attribut recherché est "current.properties.attributeName")
         * @param {string} functionName nom de la fonction contenant le nom de la variable dans un de ses arguments
         * @param {string} expression corps de la fonction <code>functionName</code> avec ses arguments sous la forme d'une chaîne de caractères
         * @return {string|null} renvoie le nom du conteneur de la variable recherchée ou bien <code>null</code>
         */
        const extractContainerName = (functionName, expression) => {
          const firstParenthesisIndex = expression.indexOf('(',0);
          const secondParenthesisIndex = expression.indexOf(')',0);

          // isole les arguments de la fonction functionName
          const functionArgsString = expression.substring(firstParenthesisIndex +1, secondParenthesisIndex);
          const functionArgsHasSeparator = functionArgsString.includes(',');
          const isCountArgsValid = functionArgsString.split(',').length === requiringFtiFunctions[functionName].length;
          if (functionArgsHasSeparator && isCountArgsValid) {

            // isole l'argument "variable source/destination" en supprimant les quotes englobantes. ex: current.properties.id_fantoir
            const functionArgsArray = functionArgsString.split(',');
            const variablePath = functionArgsArray[requiringFtiFunctions[functionName].index];
            const variablePathWithoutQuotes = variablePath.replaceAll('\'', '');

            // renvoie la 1ère partie de sourceVariablePath correspondant au nom du container. ex: current
            return variablePathWithoutQuotes.split('.').shift();
          }
          console.error('extractContainerName - Impossible d\'extraire le "conteneur" de la '
              + 'variable. La fonction ' + functionName + 'traitée possède '
              + functionArgsString.split(',').length + ' arguments alors que'
              + ' la définition de la fonction dans formFunctionUtils.requiringFtiFunctions'
              + ' en prévoit ' + requiringFtiFunctions[functionName].length);
          return null;
        }; // fin de la 1ère méthode interne de extractContainerName: extractContainerName

        /**
         * Récupère le nom de la variable de la fonction <code>functionName</code> dans la chaine de caractères de la fonction
         * @param {string} functionName nom de la fonction contenant le nom de la variable dans un de ses arguments
         * @param {string} expression corps de la fonction <code>functionName</code> avec ses arguments sous la forme d'une chaîne de caractères
         * @return {string|null} renvoie le nom de la variable recherchée ou bien <code>null</code>
         * si l'expression ne contient pas de parenthèses ou un nombre d'arguments incorrect, différent de <code>functionProcess[functionName].length</code>
         */
        const extractAttributeName = (functionName, expression) => {
          const firstParenthesisIndex = expression.indexOf('(',0);
          const secondParenthesisIndex = expression.indexOf(')',0);

          // isole les arguments de la fonction setData
          const functionArgsString = expression.substring(firstParenthesisIndex +1, secondParenthesisIndex);
          const functionArgsHasSeparator = functionArgsString.includes(',');
          const isCountArgsValid = functionArgsString.split(',').length === requiringFtiFunctions[functionName].length;

          if (functionArgsHasSeparator && isCountArgsValid) {

            // isole l'argument "variable source/destination" en supprimant les quotes englobantes. ex: current.properties.id_fantoir
            const functionArgsArray = functionArgsString.split(',');
            const variablePath = functionArgsArray[requiringFtiFunctions[functionName].index]
            const variablePathWithoutQuotes = variablePath.replaceAll('\'', '');

            // renvoie la dernière partie de sourceVariablePath correspondant au nom d'attribut. ex: id_fantoir
            return variablePathWithoutQuotes.split('.').pop();
          }
          console.error('extractAttributeName - Impossible d\'extraire le nom d\'attribut de la '
              + 'variable. La fonction ' + functionName + 'traitée possède '
              + functionArgsString.split(',').length + ' arguments alors que'
              + ' la définition de la fonction dans formFunctionUtils.requiringFtiFunctions'
              + ' en prévoit ' + requiringFtiFunctions[functionName].length);
          return null;
        }; // fin de la 2ème méthode interne de findFtiFromField: extractVariableName

        /**
         * Récupère l'objet <code>config</code> qui est une propriété de l'objet <code>field</code>
         * (champ du formulaire) correspondant à la variable source/destination de la fonction <code>functionName</code>
         * @param {string} variableName nom de la variable de la fonction <code>functionName</code>
         * (ex. 'id_fantoir' pour 'current.properties.id_fantoir')
         * @param attributeContainerName nom du conteneur de la variable de la fonction (ex. 'current' pour 'current.properties.id_fantoir')
         * @param {[object]}templateFieldsTabs onglets du formulaire contenant les champs de saisie
         * @return {object|null} objet contenant le nom de la variable source, le nom et l'uid du fti.
         * Renvoie <code>null</code> si aucun champ du formulaire ne possède le nom de la variable
         */
        const findFieldInFormTabs = (variableName, attributeContainerName, templateFieldsTabs) => {
          if (Array.isArray(templateFieldsTabs)) {
            for (const tab of templateFieldsTabs) {
              // parcours les onglets contenant les champs de formulaire
              if (tab.hasOwnProperty('fields') && Array.isArray(tab.fields)) {
                // recherche le champ source parmis les champs de l'onglet courant
                const field = tab.fields.find(tabField => tabField.hasOwnProperty('config')
                    && tabField.config.hasOwnProperty('name') && tabField.config.name === variableName
                        && tabField.config.hasOwnProperty('res') && tabField.config.res === attributeContainerName);
                if (field !== undefined) {
                  // renvoie la propriété config du field qui contient le nom de la variable source, le nom et l'uid du fti
                  return field.config;
                }
              }
            }
            console.error('findFieldInFormTabs - aucun champ du formulaire ayant les propriétés'
                + ' config.res=' + attributeContainerName + 'et config.name='
                + variableName + 'n\'a été trouvé');
          } else {
            console.error('findFieldInFormTabs - la variable des onglets du formulaire'
            +' n\'est pas un tableau');
          }
          return null;
        }; // fin de la 3ème méthode interne de findFtiFromField: findFieldInFormTabs

        /**
         * Récupère le fti source dans la configuration du champ de formulaire correspondant à la variable source/destination.<br>
         * @param {object} field champ du formulaire ayant la propriété <code>config.name</code> égal à <code>variableName</code>
         * @return {object|null} fti de la variable source ou bien <code>null</code> si la variable n'a pas été trouvée
         */
        const findFieldFti = (field) => {
          if (field !== null) {

            // on renvoie le fti ayant un uid égal à la propriété ftid du champ source du formulaire
            const fti = FeatureTypeFactory.getFeatureByUid(field.ftid);
            if (fti != null && Array.isArray(fti.attributes)) {
              return fti;
            } else if (FeatureTypeFactory.resources.featuretypes.length === 0) {
              console.error('findFieldFti - veuillez charger des composants dans l\'application au préalable');
            } else {
              console.error('findFieldFti - aucun fti n\'a été trouvé avec l\'uid fourni :'
                  + ' fti.uid = ' + field.uid);
            }
          }
          return null;
        }; // fin de la 4ème méthode interne de findFtiFromField: findFieldFti


        // ****************************
        // * findFtiFromField PROCESS *
        // ****************************

        // on recherche le nom de l'objet contenant l'attribut (attributeName)
        // ex. on recherche "current" quand l'attribut recherché est "current.properties.attributeName"
        const attributeContainerName = extractContainerName(functionName, expression);

        // on recherche le nom de l'attribut (attributeName) dans les paramètres de la fonction functionName
        const attributeName = extractAttributeName(functionName, expression);
        if (typeof attributeName === 'string' && attributeName.length > 0) {

          // on recherche le champ du formulaire de l'attribut attributeName
          let field = findFieldInFormTabs(attributeName, attributeContainerName, allTemplateTabs);

          // Si null, on recherche le champ parmi les boutons d'édition selon leur fonction finish
          if (field === null && Array.isArray(allTemplateTabs) && allTemplateTabs.length > 0
              && allTemplateTabs.every(tab => tab.hasOwnProperty('fields'))) {
            for (const tab of allTemplateTabs) {
              const finishFnFieldIndex = tab.fields.findIndex(field => Array.isArray(field.config.finish)
                  && field.config.finish.includes(expression));
              if (finishFnFieldIndex > -1) {
                field = tab.fields[finishFnFieldIndex].config;
                break;
              }
            }
          }
          if (field !== null && field.hasOwnProperty('ftid')) {

            // on récupère le fti de l'attribut représentée par le champ du formulaire
            const fti = findFieldFti(field);
            if (fti !== null) {
              return fti;
            }
          }
        }
        console.error('findFieldFti - un fti vide est renvoyé pour la fonction '
            + functionName +' nécessitant un fti pour fonctionner');
        return {};
      };

      /**
       * Défini l'identifiant utilisé lors de l'upload d'un ou plusieurs fichiers attachés
       * @param fieldData configuration du champ de type gcattachment(s)
       * @return {string} identifiant servant du nom de sous-répertoire UPLOAD dans lequels seront déposés les fichiers attachés
       */
      const setAttachmentProcessId = (fieldData) => {
        if (!fieldData.attachmentId) {
          fieldData.attachmentId = gaJsUtils.guid();
        }
        return fieldData.attachmentId;
      };

      /**
       * Supprime les quotes englobantes des arguments des fonctions de formulaire
       * Suite au ticket KIS-3538, on doit supprimer les quotes englobantes à l'évaluation.
       *
       * @param args arguments d'une fonction de formulaire
       * @return {*} arguments sans les quotes englobantes
       */
      const deleteSurrondingQuotes = (args) => {
        for (let i = 0; i < args.length; i++) {
          let argument = args[i];
          if (argument.startsWith('\'')) {
            argument = argument.slice(1);  // Supprime la 1ère quote
          }
          if (argument.endsWith('\'')) {
            argument = argument.slice(0, -1);  // Supprime la dernière quote
          }
          args[i] = argument;  // Réinjecte dans l'array modifié
        }
        return args;
      };

      return {
        hasTabIndex: hasTabIndex,
        getTabIndex: getTabIndex,
        getTabHolder: getTabHolder,
        getDataValueByKey: getDataValueByKey,
        someFunctionRequireFti: someFunctionRequireFti,
        findRequiredFtis: findRequiredFtis,
        setAttachmentProcessId: setAttachmentProcessId,
        deleteSurrondingQuotes: deleteSurrondingQuotes
      };
    };
    this.$get.$inject = ['FeatureTypeFactory', 'gaJsUtils'];
  };
  return formFunctionUtils;
});
