'use strict';
define(function () {
  /**
   * Fonction "setCurrentDate" pour affecter la date et heure du jour.<br>
   * <a href="https://altereo-informatique.atlassian.net/browse/KIS-3209">KIS-3209</a><ul><li>
   * Dans le cas où l’attribut possède une valeur nulle de date alors l’attribut doit être mis à jour sans qu’il y ait besoin d’activer l’option.</li><li>
   * Dans le cas où l’attribut possède une valeur non nulle de date alors l’attribut est mis à jour avec la date courante que si l’option est activée.</li></ul>
   */
  const setCurrentDateFunction = function() {
    this.$get = ($q, $filter, gaJsUtils) => {

      /**
       * Lance le processus d'insertion de date dans un tableau de feature
       * @param {object[]} features tableau d'objets
       * @param {string} attributeName nom de l'attribut date à alimenter
       * @param {boolean} eraseExistingDate option qui a la valeur <code>true</code> pour forcer le remplacement d’une date existante par la date courante.
       */
      const setFeaturesCurrentDate = (features, attributeName, eraseExistingDate) => {
        for (const feature of features) {
          if (!gaJsUtils.notNullAndDefined(feature.properties)) {
            console.error(`setCurrentDate - l'objet ${feature} n'est pas un objet geojson, le champ ${attributeName} ne peut pas être alimenté`);
            break;
          }
          const destination = feature.properties;
          // Vérifier si l'attribut est null/undefined ou si l'option "Remplacer la valeur" est cochée
          if (!destination[attributeName] || eraseExistingDate) {
            destination[attributeName] = new Date();
          }
        }
      };

      return  {
        getParametersDesc: () => {
          return [
            [0, 'variable'],
            [1, 'string'],
            [2, 'boolean']
          ];
        },

        /**
         * Alimente un attribut avec la date courante du système (KIS-3209 | [DEA - IS] : nouvelle fonction "setCurrentDate" pour affecter la date et heure du jour)
         * @param {object} variables objet res qui contient les variables du formulaire courant
         * @param {string} destVarName nom de la variable à alimenter. Contenu du champ "Destination" de la définition de la fonction (ex. 'current', 'selection_cible'...)
         * @param {string} attributeName contenu du champ "Attribut" de la définition de la fonction
         * @param {string} erase option qui a la valeur <code>"tab-true"</code> pour forcer le remplacement d’une date existante par la date courante.
         * @return {Promise}
         */
        setCurrentDate: (variables, destVarName, attributeName, erase) => {
          const defer = $q.defer();
          const eraseExistingDate = erase === 'tab-true';

          // isole la variable de destination (ex. res.current, res.selection_cible, res.geometrie_cible...)
          const destinationVar = variables[destVarName];

          // la variable de destination peut être soit de type Array, soit de type FeatureCollection, soit de type Feature.
          if (Array.isArray(destinationVar)) {

            // destinationVar est un tableau de features (ex. données issues d'une sélection d'objets)
            setFeaturesCurrentDate(destinationVar, attributeName, eraseExistingDate);

          } else if (destinationVar.hasOwnProperty('type') && typeof destinationVar.type === 'string'
              && destinationVar.type === 'FeatureCollection'
              && Array.isArray(destinationVar.features) && destinationVar.features.length > 0) {

            // destinationVar est une FeatureCollection (ex. données issues d'une datatable)
            setFeaturesCurrentDate(destinationVar.features, attributeName, eraseExistingDate);

          } else {

            // destinationVar est une feature (ex. intervention simple: 'current')
            setFeaturesCurrentDate([destinationVar], attributeName, eraseExistingDate);

          }

          defer.resolve();
          return defer.promise;
        }
      };
    };

    this.$get.$inject = ['$q', '$filter', 'gaJsUtils'];
  };
  return setCurrentDateFunction;
});
