'use strict';
define(() => {
  const styleconverter = (FeatureTypeFactory, StyleConverterFactory, $filter) => {
    return {
      templateUrl: 'js/XG/modules/model/views/directives/styleconverter.html',
      restrict: 'EA',
      link: function (scope) {

        /**
         * A la selection d'un format d'entrée,
         * affiche les labels adaptés au format d'entrée
         * et met à jour les formats de sortie disponibles
         */
        scope.onInputFormatChange = () => {
          if (scope.conversionData.inputFormat === 'Mapserver') {
            scope.checkFileWarning = $filter('translate')('styleConverter.mapserverWarning');
          } else if (scope.checkFileWarning) {
            scope.checkFileWarning = null;
          }
          if (scope.conversionData.file) {
            scope.conversionData.file = null;
          }
          updateOutputFormats();
        };

        /**
         * Met à jour les formats de sortie disponibles
         */
        const updateOutputFormats = () => {
          scope.inputAcceptedExtension = StyleConverterFactory.apiCapabilities[scope.conversionData.inputFormat].extension;
          scope.outputFormats = StyleConverterFactory.apiCapabilities[scope.conversionData.inputFormat].export;
        };

        /**
         * Simule un clic sur l'input-file au clic sur le bouton d'upload de fichier
         * @param event
         */
        scope.openFileDialog = (event) => {
          const button = event.currentTarget;
          const container = button.parentNode;
          if (container) {
            const fileInput = container.querySelector('input[type="file"]');
            if (fileInput) {
              fileInput.click();
            }
          }
        };

        /**
         * Gère le changement de fichier dans l'input-file.
         * Vérifie si le fichier est un SLD et lance la vérification du fichier SLD si nécessaire.
         * @param input
         */
        scope.onInputFileChange = (input) => {
          let file;
          if (input.files && input.files[0]) {
            file = input.files[0];
          }
          if (scope.conversionData.inputFormat === 'SLD') {
            checkSldFile(file);
          } else {
            scope.conversionData.file = file;
            scope.checkFileError = null;
          }
          // Déclencher le digest cycle pour mettre à jour la vue
          scope.$apply();
        };

        /**
         * Vérifie si le fichier SLD est valide.
         * @param file
         */
        const checkSldFile = (file) => {
          scope.checkSldFileProcessing = true;
          StyleConverterFactory.checkSldFile(file).then(
            (response) => {
              if (typeof response.data === 'boolean' && response.data) {
                scope.conversionData.file = file;
              } else {
                scope.checkFileError = $filter('translate')('styleConverter.sldInvalid');
              }
            },
            (error) => {
              scope.checkFileError = $filter('translate')('styleConverter.sldCheckError');
              console.error('Error checking SLD file:', error);
            }
          ).finally(() => {
            scope.checkSldFileProcessing = false;
          });
        };

        /**
         *  A la sélection d'un composant lorsque le format d'entrée est SLD
         */
        scope.onLayerChange = () => {
          if (scope.selectedLayer) {
            scope.conversionData.sldName = null;
          }
        };

        /**
         * Suprime le fichier chargé
         */
        scope.deleteUploadedFile = () => {
          scope.conversionData.file = null;
          scope.checkFileError = null;
        };

        /**
         * Vérifie si le formulaire de conversion est valide
         * @return {boolean}
         */
        scope.isFormValid = function() {
          // Vérifier que inputFormat et outputFormat sont renseignés
          if (!scope.conversionData.inputFormat || !scope.conversionData.outputFormat) {
            return false;
          }

          return scope.isFileOrSldValid();
        };

        /**
         * Vérifie si le fichier ou le SLD est valide
         * @return {boolean}
         */
        scope.isFileOrSldValid = function() {
          if (scope.conversionData.inputFormat === 'SLD') {
            // Pour SLD : soit un fichier chargé, soit sldName existe
            return !!scope.conversionData.file || !!scope.conversionData.sldName;
          } else {
            // Pour les autres formats : un fichier doit être chargé
            return !!scope.conversionData.file;
          }
        };

        /**
         * Convertit le style en fonction des données de conversion
         */
        scope.convert = () => {
          scope.isConverting = true;
          StyleConverterFactory.convert(scope.conversionData).then(
              (response) => {
                scope.isConverting = false;
                // Créer un URL object à partir du blob reçu
                const blob = new Blob([response.data]);
                const url = window.URL.createObjectURL(blob);

                // Créer un lien temporaire pour déclencher le téléchargement
                const a = document.createElement('a');
                a.href = url;
                // Récupérer le nom du fichier depuis les headers de la réponse
                const contentDisposition = response.headers('content-disposition');
                a.download = contentDisposition ? contentDisposition.split(
                    'filename=')[1].replace(/"/g, '') : 'style.txt';

                // Déclencher le téléchargement
                document.body.appendChild(a);
                a.click();

                // Nettoyer
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                scope.conversionData.file = null;
              },
              (error) => {
                scope.isConverting = false;
                if (typeof error.data === 'string') {
                  console.error(error.data);
                }
                require('toastr').error($filter('translate')('styleConverter.errorConvertingStyle'));
              });
        };

        /**
         * Initialise la directive
         */
        const initDirective = () => {
          FeatureTypeFactory.get(false).then(res => {
            scope.featureTypes = res.filter(ft => Array.isArray(ft.styles) && ft.styles.length > 0);
          });

          scope.inputLabel = $filter('translate')('styleconverter.input_file');

          scope.conversionData = {
            inputFormat: 'SLD',
            outputFormat: 'QML',
          };

          scope.inputFormats = Object.keys(StyleConverterFactory.apiCapabilities);
          updateOutputFormats();
        };

        initDirective();
      },
    };
  };
  styleconverter.$inject = ['FeatureTypeFactory', 'StyleConverterFactory', '$filter'];
  return styleconverter;
});