'use strict';
define(function() {
  /**
   * Class : RightsFactory
   * Factory WebServices
   */
  var RightsFactory = function($http) {
    var resources = {
      users: null,
      roles: null,
      groups: null,
    };

    /**
     * Function: getallusers
     */
    function getallusers() {
      var promise = $http.get('/services/users/listusers?f=json');
      promise.then(function(res) {
        resources.users = res.data;
      });
      return promise;
    }
    /**
     * Function: getallroles
     */
    function getallroles() {
      var promise = $http.get('/services/users/listroles?f=json');
      promise.then(function(res) {
        resources.roles = res.data;
      });
      return promise;
    }
    /**
     * Function: getallgroups
     */
    function getallgroups() {
      var promise = $http.get('/services/users/listgroups?f=json');
      promise.then(function(res) {
        resources.groups = res.data;
      });
      return promise;
    }

    /**
     * Function: removegroup
     */
    function removegroup(token, id, callback, callbackerror) {
      $http
        .get(
          '/services/users/removegroup?f=json&' +
            'id=' +
            id +
            '&' +
            'token=' +
            token
        )
        .then(function (data) {
          //console.log(data);
          if (data.CLASSTYPE == 'Error') {
            require('toastr').error('Error ' + data.message);
            callbackerror(data);
            return data;
          }
          callback(data);
          return data;
        })
        .catch(function (data, status) {
          require('toastr').error('Error' + data + ' ' + status);
          callbackerror(data);
          return status;
        });
    }

    /**
     * Function: removerole
     */
    function removerole(token, id, callback, callbackerror) {
      $http
        .get(
          '/services/users/removerole?f=json&' +
            'id=' +
            id +
            '&' +
            'token=' +
            token
        )
        .then(function (data) {
          //console.log(data);
          if (data.CLASSTYPE == 'Error') {
            require('toastr').error('Error ' + data.message);
            callbackerror(data);
            return data;
          }
          callback(data);
          return data;
        })
        .catch(function (data, status) {
          require('toastr').error('Error' + data + ' ' + status);
          callbackerror(data);
          return status;
        });
    }

    /**
     * Function: removeuser
     */
    function removeuser(id, callback, callbackerror) {
      $http
        .get('/services/users/removeuser?f=json&' + 'id=' + id)
        .then(function (data) {
          //console.log(data);
          if (data.CLASSTYPE == 'Error') {
            require('toastr').error('Error ' + data.message);
            callbackerror(data);
            return data;
          }
          callback(data);
          return data;
        })
        .catch(function (data, status) {
          require('toastr').error('Error' + data + ' ' + status);
          callbackerror(data);
          return status;
        });
    }

    /**
     * Function: adduser
     */
    function adduser(user, pass) {
      var promise = $http.post(
        '/services/users/adduser?f=json&' + 'pass=' + pass,
        user
      );
      promise.then(function(res) {
        //console.log(res.data);
        //console.log(resources);
        resources.users.push(res.data);
      });
      return promise;
    }

    /**
     * Function: addrole
     */
    function addrole(senddata) {
      var promise = $http.post('/services/users/addrole?f=json', senddata);
      promise.then(function(res) {
        resources.roles.push(res.data);
      });
      return promise;
    }

    /**
     * Function: addgroup
     */
    function addgroup(senddata) {
      var promise = $http.post('/services/users/addgroup?f=json', senddata);
      promise.then(function(res) {
        resources.groups.push(res.data);
      });
      return promise;
    }

    const roleHasReadRightsOnAttribute = (role, ftiUid, attributeName) => {
      if (role.authorizations) {
        // On cherche notre FTI
        for (const authorization of role.authorizations) {
          if (authorization.item === ftiUid) {
            // On cherche notre attribut
            for (const subItem of authorization.subItems) {
              if(subItem.item === attributeName) {
                if (subItem.read) {
                  return true;
                }
              }
            }
          }
        }
      }
      return false;
    };

    /**
     * Cherche si un utilisateur a les droits de lecture sur un attribut d'une feature
     * @param user utilisateur
     * @param ftiUid uid de la feature
     * @param attributeName nom de l'attribut
     * @returns {boolean}
     */
    const isAllowedToReadAttributesInFeature = (user, ftiUid, attributeName) => {
      if (user.admin) {
        return true;
      }

      // Roles
      for (const role of user.roles) {
        if (roleHasReadRightsOnAttribute(role, ftiUid, attributeName)) {
          return true;
        }
      }
      // Groupes
      for (const group of user.groups) {
        for (const role of group.roles) {
          if (roleHasReadRightsOnAttribute(role, ftiUid, attributeName)) {
            return true;
          }
        }
      }
      return false;
    }

    const roleHasWriteRightsOnAttribute = (role, ftiUid, attributeName) => {
      if (role.authorizations) {
        // On cherche notre FTI
        for (const authorization of role.authorizations) {
          if (authorization.item === ftiUid) {
            // On cherche notre attribut
            for (const subItem of authorization.subItems) {
              if(subItem.item === attributeName) {
                if (subItem.write) {
                  return true;
                }
              }
            }
          }
        }
      }
      return false;
    };

    /**
     * Cherche si un utilisateur a les droits en écriture sur un attribut d'une feature
     * @param user utilisateur
     * @param ftiUid uid de la feature
     * @param attributeName nom de l'attribut
     * @returns {boolean}
     */
    const isAllowedToWriteAttributesInFeature = (user, ftiUid, attributeName) => {
      if (user.admin) {
        return true;
      }
      // Roles
      for (const role of user.roles) {
        if (roleHasWriteRightsOnAttribute(role, ftiUid, attributeName)) {
          return true;
        }
      }
      // Groupes
      for (const group of user.groups) {
        for (const role of group.roles) {
          if (roleHasWriteRightsOnAttribute(role, ftiUid, attributeName)) {
            return true;
          }
        }
      }
      return false;
    };

    /**
     * Vérifie si le composant (fti) possède un rôle commun avec l'utilisateur
     * et si le rôle possède un droit de lecture ou d'écriture sur le composant
     * @param {object} fti objet du composant pour lequel on veut savoir
     * si l'utilisateur a un droit en écriture
     * @param {boolean} isAdmin est true si l'utilisateur est admin
     * (issu de $rootScope.xgos.isroot/isadmin)
     * @param {object} user objet utilisateur connecté issu de $rootScope.xgos.user
     * @param {boolean} checkReadRightOnly (false par défaut) est true si
     * on donne l'accès au fti en se basant sur le droit en lecture.
     * False quand on se base sur le droit en écriture
     * @return {boolean} true si l'utilisateur détient un rôle contenant
     * un droit en écriture sur le composant
     */
    const isAllowedToReadOrWriteFeatureType = (fti, isAdmin, user, checkReadRightOnly = false) => {

      const isRoleAllowedToReadOrWriteFeatureType = (roles) => {
        for (const role of roles) {
          if (Array.isArray(role.authorizations)) {
            const userComponent = role.authorizations.find(auth => auth.item === fti.uid);
            if (userComponent) {
              const rightToCheck = checkReadRightOnly ? userComponent.read
                : userComponent.write;
              if (typeof rightToCheck === 'boolean' && rightToCheck) {
                return true;
              }
            }
          }
        }
        return false;
      };

      const createGroupRoles = (user) => {
        const grouproles = [];
        for (const group of user.groups) {
          if (Array.isArray(group.roles)) {
            for (const role of group.roles) {
              if (!grouproles.find(grprole => grprole.name === role.name)) {
                grouproles.push(role);
              }
            }
          }
        }
        return grouproles;
      };

      if (fti && fti.uid) {
        if (isAdmin) {
          return true;
        } else if (user) {
          if (Array.isArray(user.roles) && isRoleAllowedToReadOrWriteFeatureType(user.roles)) {
            return true;
          }
          if (Array.isArray(user.groups)) {
            const grouproles = createGroupRoles(user);
            if (grouproles.length > 0 && isRoleAllowedToReadOrWriteFeatureType(grouproles)) {
              return true;
            }
          }
        }
      }
      return false;

    };

    const roleHasWriteRightsOnFeature = (role, ftiUid) => {
      if (role.authorizations) {
        // On cherche notre FTI
        for (const authorization of role.authorizations) {
          if (authorization.item === ftiUid) {
            return authorization.write;
          }
        }
      }
      return false;
    };

    /** Cherche si l'utilisateur a les droits en écriture sur une feature
     * Cherche si un
     * @param user
     * @param ftiUid
     * @returns {boolean}
     */
    const isAllowedToWriteFeature = (user, ftiUid) => {
      if (user.admin) {
        return true;
      }
      // Roles
      for (const role of user.roles) {
        if (roleHasWriteRightsOnFeature(role, ftiUid)) {
          return true;
        }
      }
      // Groupes
      for (const group of user.groups) {
        for (const role of group.roles) {
          if (roleHasWriteRightsOnFeature(role, ftiUid)) {
            return true;
          }
        }
      }
      return false;
    };

    /**
     * Récupère les attributs sur lesquels l'utilisateur a un droit en écriture ou lecture.<br>
     * Le droit à vérifier (lecture ou écriture) est fourni en paramètre.
     * @param {object} user utilisateur dont on vérifie les droits
     * @param {object} fti featuretype contenant les attributs à récupérer
     * @param {boolean} isAdmin est true si xgos.isadmin/isroot = true
     * @param {boolean} checkReadRightOnly (optional) est true si on souhaite
     * vérifier les droits en lecture de l'utilisateur sur l'attribut.
     * On vérifie le droit en écriture si false. Le paramètre est false par défaut.
     * @return {object[]} tableau des attributs du featuretype dont l'utilisateur a le droit
     * en lecture ou écriture en fonction du paramètre <code>checkReadRightOnly</code>
     */
    const getUserWriteOrReadRightsAttributes = (user, fti, isAdmin, checkReadRightOnly = false) => {
      if (isAdmin) {
        return fti.attributes;
      }

      const userRights = {};
      for (const role of user.roles) {
        for (const authorization of role.authorizations) {
          if (authorization.item === fti.uid) {
            for (const subItem of authorization.subItems) {
              userRights[subItem.item] = {
                read: userRights[subItem.item] === undefined || !userRights[subItem.item].read
                          ? subItem.read : true,
                  write: userRights[subItem.item] === undefined || !userRights[subItem.item].write
                          ? subItem.write : true,
              };
            }
          }
        }
      }
      for (const group of user.groups) {
        for (const groupRole of group.roles) {
          for (const authorization of groupRole.authorizations) {
            if (authorization.item === fti.uid) {
              for (const subItem of authorization.subItems) {
                userRights[subItem.item] = {
                  read: userRights[subItem.item] === undefined || !userRights[subItem.item].read
                          ? subItem.read : true,
                  write: userRights[subItem.item] === undefined || !userRights[subItem.item].write
                          ? subItem.write : true,
                };
              }
            }
          }
        }
      }

      const authorizedAttrs = [];
      for (const attr of fti.attributes) {
        if (userRights[attr.name] !== undefined &&
            (userRights[attr.name].write === true ||
              (checkReadRightOnly && userRights[attr.name].read === true))) {
          authorizedAttrs.push(attr);
        }
      }
      return authorizedAttrs;
    };

    return {
      RightsFactory: RightsFactory,

      resources: resources,

      getallroles: getallroles,
      getallusers: getallusers,
      removegroup: removegroup,
      addrole: addrole,
      getallgroups: getallgroups,
      addgroup: addgroup,
      removerole: removerole,
      removeuser: removeuser,
      adduser: adduser,
      isAllowedToReadAttributesInFeature: isAllowedToReadAttributesInFeature,
      isAllowedToWriteAttributesInFeature: isAllowedToWriteAttributesInFeature,
      isAllowedToWriteFeature: isAllowedToWriteFeature,
      isAllowedToReadOrWriteFeatureType: isAllowedToReadOrWriteFeatureType,
      getUserWriteOrReadRightsAttributes : getUserWriteOrReadRightsAttributes
    };
  };
  RightsFactory.$inject = ['$http'];
  return RightsFactory;
});
