'use strict';
define(() => {
  class importgpkgtwidget {
    constructor(extendedNgDialog, $filter, gaDomUtils, EditFactory,
        FeatureTypeFactory, $rootScope, ImportExportFactory) {
      return {
        templateUrl:
            'js/XG/widgets/mapapp/importgpkg/views/importgpkgwidget.html',
        restrict: 'AE',
        link: scope => {
          // enable importFile() to filter between dwg and dxf
          scope.format = '.gpkg';

          /**
           * Enregistre les feautres dans la couche de destination
           * @param layer couche source contenant les features à sauvegarder
           * @param applyRules indique si les règles de calcul attributaire doivent être appliquées
           */
          const saveGpkg = (layer, applyRules) => {
            gaDomUtils.showGlobalLoader();
            const geojson = layer.geojson;
            const defaultLayer = angular.copy(geojson);
            try {
              for (const feature of geojson.features) {
                updateFeatureProperties(layer, feature);
              }
              if (layer.importcfg.maj.maj_component) {
                updateFeatures(layer, geojson, applyRules, defaultLayer);
              } else {
                addFeatures(layer, geojson, applyRules, defaultLayer);
              }
            } catch (e) {
              console.error(e.message);
              if (defaultLayer) {
                layer.geojson = angular.copy(defaultLayer);
              }
              gaDomUtils.hideGlobalLoader();
              require('toastr').error(
                  $filter('translate')('importdiverswidget.copyerror') + layer.name
              );
            }
          };

          /**
           * Gère l'ouverture de la fenetre du mapping de l'import
           * @param layer couche source à importer
           * @param childScope scope du composant enfant dxfwidget contenant la configuration de l'import
           */
          scope.copyFn = (layer, childScope) => {
            if (layer.importcfg) {
              layer.importcfg.showPurgeLayer = true;
            }

            // transmission de la variable par input jusqu'au petit-fils gcimportfeatures
            scope.transferSuccessful = {value: false}

            // liaison entre les méthodes de importgpkgwidget
            // et les méthodes du scope de importdxfwidget qui sont transmises à gcimportfeatures
            childScope.saveGpkg = saveGpkg;
            childScope.downloadGpkgJournal = downloadJournal;

            // la popup doit utiliser le scope du composant enfant importdxfwidget
            // car celui-ci contient la configuration de l'import
            extendedNgDialog.open({
              template:
                  'js/XG/widgets/mapapp/importgpkg/views/dialogs/importgpkgdialog.html',
              title: $filter('translate')('importdxfwidget.copytitle') + layer.name,
              draggable: true,
              className:
                  'ngdialog-theme-plain width800 nopadding miniclose',
              closeByDocument: false,
              scope: childScope,
            });
          };

          /**
           * Gère les champs de jointure pour la mise à jour des features
           * lorsque la case "Mise à jour attributaire" est cochée
           * @param maj configuration de la mise à jour
           * @param feature feature à mettre à jour
           * @return {{}} nouvelles propriétés du feature
           */
          const handleMajComponent = (maj, feature) => {
            const propnew = {};
            if (maj.maj_component && Object.keys(maj.attribut).length > 0) {
              if (maj.attribut !== 'fid') {
                propnew[maj.value.name] = feature.properties[maj.attribut];
              } else {
                feature.id = feature.properties[maj.attribut];
              }
            }
            return propnew;
          };

          /**
           * Gère uniquement les liaisons entre champs source et destination
           * @param maj configuration de la mise à jour
           * @param layer couche source à importer
           * @param feature feature à mettre à jour
           * @return {{}} nouvelles propriétés du feature
           */
          const handleLiaisonsOnly = (maj, layer, feature) => {
            const propnew = {};
            for (const [key, value] of Object.entries(layer.importcfg.liaisons)) {
              if ((maj.maj_component && Object.keys(maj.attribut).length > 0 && key !== maj.attribut) || !maj.maj_component) {
                propnew[value] = feature.properties[key];
              }
            }
            return propnew;
          };

          /**
           * Gère uniquement les valeurs par défaut
           * @param maj configuration de la mise à jour
           * @param layer couche source à importer
           * @return {{}} nouvelles propriétés du feature
           */
          const handleValuesOnly = (maj, layer) => {
            const propnew = {};
            for (const [key, value] of Object.entries(layer.importcfg.values)) {
              if ((maj.maj_component && Object.keys(maj.attribut).length > 0 && key !== maj.attribut) || !maj.maj_component) {
                propnew[key] = value;
              }
            }
            return propnew;
          };

          /**
           * Gère à la fois les liaisons et les valeurs par défaut
           * @param maj configuration de la mise à jour
           * @param layer couche source à importer
           * @param feature feature à mettre à jour
           * @return {{}} nouvelles propriétés du feature
           */
          const handleLiaisonsAndValues = (maj, layer, feature) => {
            let propnew = {};
            // Traitement des valeurs par défaut
            for (const [key, value] of Object.entries(layer.importcfg.values)) {
              if ((maj.maj_component && Object.keys(maj.value).length > 0 && key !== maj.value.name) || !maj.maj_component) {
                propnew[key] = value;
              }
            }

            // Traitement des liaisons
            for (const [key, value] of Object.entries(layer.importcfg.liaisons)) {
              let newVal = feature.properties[key];
              const overrideDefault = Object.keys(propnew).indexOf(key) > -1 &&
                  Object.keys(layer.importcfg.values).indexOf(key) > -1;

              if (maj.maj_component && Object.keys(maj.attribut).length > 0 && key !== maj.attribut) {
                propnew[value] = newVal;
              } else if ((Object.keys(propnew).indexOf(key) === -1 ||
                  (propnew[value] === null && newVal !== null) ||
                  (overrideDefault && newVal !== null)) &&
                  !maj.maj_component) {
                propnew[value] = newVal;
              }
            }
            return propnew;
          };

          /**
           * Mise à jour des propriétés d'un feature selon la configuration de l'import
           * @param layer couche source
           * @param feature feature à mettre à jour
           */
          const updateFeatureProperties = (layer, feature) => {
            let propnew = {};
            const values = Object.keys(layer.importcfg.values);
            const liaisons = Object.keys(layer.importcfg.liaisons);
            const maj = layer.importcfg.maj;

            propnew = handleMajComponent(maj, feature);

            if (liaisons.length === 0 && values.length === 0) {
              // si aucun champ lié et aucun champ avec valeur par défaut
              propnew = {};
            } else if (liaisons.length > 0 && values.length === 0) {
              // si au moins 1 champ lié et aucun champ avec valeur par défaut
              propnew = handleLiaisonsOnly(maj, layer, feature);
            } else if (liaisons.length === 0 && values.length > 0) {
              propnew = handleValuesOnly(maj, layer);
            } else {
              // si au moins 1 champ lié et au moins 1 champ avec valeur par défaut
              propnew = handleLiaisonsAndValues(maj, layer, feature);
            }
            feature.properties = propnew;
          };

          /**
           * Ajoute les features dans la couche de destination
           * @param layer couche source
           * @param geojson geojson de la couche source
           * @param applyRules indique si les règles de calcul attributaire doivent tre appliquées
           * @param defaultLayer couche d'origine sauvegardée en cas d'erreur
           */
          const addFeatures = (layer, geojson, applyRules, defaultLayer) => {
            const purgeLayer = layer.importcfg.purgeLayer && layer.importcfg.purgeLayer.value === true;
            // ajoute les lignes du CSV dans le composant layer.importcfg.ftid
            EditFactory.add(
                layer.importcfg.ftid,
                geojson,
                scope.map.getView().getProjection().getCode(),
                'false',
                applyRules,
                false,
                purgeLayer
            ).then(
                (result) => {
                  handleSuccess(result, layer, defaultLayer)
                },
                (result) => {
                  handleError(result, defaultLayer);
                }
            );
          };

          /**
           * Met à jour les features existants dans la couche de destination
           * @param layer couche source
           * @param geojson geojson de la couche source
           * @param applyRules indique si les règles de calcul attributaire doivent tre appliquées
           * @param defaultLayer couche d'origine sauvegardée en cas d'erreur
           */
          const updateFeatures = (layer, geojson, applyRules, defaultLayer) => {
            const purgeLayer = layer.importcfg.purgeLayer && layer.importcfg.purgeLayer.value === true;
            const fti = FeatureTypeFactory.getFeatureByUid(
                layer.importcfg.ftid
            );
            // mise à jour des objets du composant layer.importcfg.ftid basé sur geojson.features[x].id
            const sendata = {
              attribut: layer.importcfg.maj.value.name,
              geojson: geojson,
            };
            EditFactory.specialUpdate(
                layer.importcfg.ftid,
                sendata,
                fti.srid,
                'false',
                applyRules,
                purgeLayer,
                false
            ).then(
                (result) => {
                  handleSuccess(result, layer, defaultLayer, true)
                },
                (result) => {
                  handleError(result, defaultLayer);
                }
            );
          };

          /**
           * Gère le succès de l'import / mise à jour
           * @param result résultat de l'appel à l'API
           * @param layer couche source
           * @param defaultLayer sauvegarde de la couche d'origine
           * @param isUpdate true si c'est une mise à jour, false si c'est un ajout
           */
          const handleSuccess = (result, layer, defaultLayer, isUpdate = false) => {
            const processedFeatures = isUpdate ? result.data.update : result.data.create;
            if (
                result.data.errors.length === 0 ||
                processedFeatures.length === layer.geojson.features.length
            ) {

              // met à jour les styles,cql_filter,opacity... esri et geoserver
              $rootScope.$broadcast(
                  'gcOperationalLayerChange',
                  '',
                  'applyall'
              );

              scope.transferSuccessful = {value: true};
              scope.journalUrl = result.data.journalUrl;

              gaDomUtils.hideGlobalLoader();
              require('toastr').success(
                  ($filter('translate')('importGeopackage.success')).replace(
                      '$1', processedFeatures.length));
            } else {
              layer.geojson = angular.copy(defaultLayer);
              gaDomUtils.hideGlobalLoader();
              require('toastr').error(
                  $filter('translate')('importGeopackage.error').replace(
                      '$1', result.data.errors.length));
            }
          };

          /**
           * Gère les erreurs lors de l'import / mise à jour
           * @param result résultat de l'appel à l'API
           * @param layer couche source
           */
          const handleError = (result, layer) => {
            gaDomUtils.hideGlobalLoader();
            if (
                result.data &&
                result.data.code === 403 &&
                result.data.CLASSTYPE === 'Error' &&
                result.data.message &&
                result.data.details
            ) {
              gcRestrictionProvider.showDetailsErrorMessage(result);
            } else {
              require('toastr').error(
                  $filter('translate')('importdiverswidget.copyerror') +
                  layer.name
              );
            }
          };

          /**
           * Utilise l'API Fetch pour télécharger le fichier
           * Crée un blob à partir de la réponse, ce qui préserve l'encodage
           * Déclenche le téléchargement via un élément <a> temporaire
           * Nettoie les ressources après le téléchargement
           * Gère les erreurs avec un message utilisateur
           */
          const downloadJournal = () => {
            ImportExportFactory.downloadJournal(scope.journalUrl);
          };
        },
      };
    }
  }

  importgpkgtwidget.$inject = ['extendedNgDialog', '$filter', 'gaDomUtils',
    'EditFactory', 'FeatureTypeFactory', '$rootScope', 'ImportExportFactory'];
  return importgpkgtwidget;
});
