'use strict';
define(() => {

  /**
   * Widget Import GPX
   *
   * Import des points, lignes et polygones
   * Utilisation des popups du widget import DXF
   */
  class importgpxwidget {
    constructor(
      importgpxservice,
      ImportExportFactory,
      FeatureTypeFactory,
      FeatureAttachmentFactory,
      gclayers,
      $filter,
      extendedNgDialog,
      $timeout,
      EditFactory,
      $rootScope,
      gaDomUtils,
      gcStyleFactory,
      SelectManager,
      gcPopup,
      gcRestrictionProvider,
      gcInteractions
    ) {
      return {
        templateUrl:
            'js/XG/widgets/mapapp/importgpx/views/importgpxwidget.html',
        restrict: 'AE',
        link: function (scope) {
          const FORMAT_DATE = 'yyyy-MM-ddTHH:mm:ss.sssZ';

          // taille de l'extent quand la layer ne contient qu'un seul point (évite plantage openlayers)
          // modifier cette valeur pour modifier la taille de l'extent
          const DISTANCE = 50;

          // extensions considérées pour les attachements
          const allowedAttachments = /(\.jpg|\.jpeg|\.png|\.gif|\.tif|\.txt|\.doc|\.ods|\.odt|\.xls|\.docx|\.xlsx|\.xml|\.pdf)$/i;

          // projection
          const mapProjection = scope.map.getView().getProjection().getCode();

          const parser = new ol.format.GeoJSON();

          // gestion spinner
          scope.waitImport = false;

          // désactivation au démarrage
          scope.selectionisactive = false;

          // onglet actif
          scope.gpxtabs = {
            activeTab: 0
          };

          // fournit l'id du processus lors de la sauvegarde
          let processId;

          // utile pour masquer le spinner à la fin de l'import
          // @see parsePhotoToGeoJSONAndBuildImportLayer
          let hasGPX;

          // popup de sélection des features
          let pop;

          // popup de la table de correspondances
          let dialog;

          // stocke le geojson source pour le restaurer en cas d'erreur de mappage vers le fti
          let defaultLayer;

          /**
           * Fonction qui attribue le style à chaque feature.<br>
           * Attribue un marker différent aux photos
           * @param feature objet openlayers dont le style est à définir
           * @returns {ol.style.Style} style appliqué à la couche d'import KIS
           */
          let myStyle = function (feature) {
            if (feature.getGeometry().getType() === 'Point'
                && feature.getProperties().type && feature.getProperties().type
                === 'photo-tag') {
              return pictureStyle;
            } else {
              return defaultStyle;
            }
          }

          // styles des objets
          let pictureStyle = new ol.style.Style({
            image: new ol.style.Icon(/** @type {olx.style.IconOptions} */ ({
              anchor: [0.5, 1.0],
              size: [23, 32],
              anchorXUnits: 'fraction',
              anchorYUnits: 'fraction',
              src: 'img/widget/gpx/pic_marker.png'
            }))
          });

          // styles des objets
          let defaultStyle = new ol.style.Style({
            fill: new ol.style.Fill({
              color: 'rgba(255, 255, 255, 0.6)',
            }),
            stroke: new ol.style.Stroke({
              color: '#ff0000',
              width: 2,
            }),
            image: new ol.style.Icon(({
              anchor: [0.5, 1.0],
              size: [23, 32],
              anchorXUnits: 'fraction',
              anchorYUnits: 'fraction',
              src: 'img/widget/gpx/gpx_marker.png'
            }))
          });

          /**
           * Création de l'objet contenant les onglets et leur propriétés<br><ul><li>
           * title: nom de l'onglet affiché</li><li>
           * disabled: état courant de l'onglet</li></ul>
           */
          function setTabs() {
            scope.tabs = [
              {
                title: $filter('translate')('importexportwidget.import'),
                disabled: false
              },
              {
                title: $filter('translate')('importexportwidget.dataset'),
                disabled: true
              },
            ];
          }

          // fonction détachée pour pouvoir être exécutée après navigation
          // @see resetViewAfterNavigation
          setTabs();

          // restaure la liste des couches après navigation
          if (importgpxservice.layers && importgpxservice.layers.length > 0) {
            resetViewAfterNavigation();
          }

          /**
           * Au clic sur le bouton "Importer"<br>
           * Récupère les fichiers en string du serveur<br>
           * Parse ici en xml pour convertir en JSON et construire les layers<br>
           * Il y a autant de layer que de type de géométrie présent parmi "Point","Linestring","Polygon"<br>
           * La librairie xml->geojson est en JS (Gdal n'accepte pas les GPX avec plusieurs types de géométrie)
           * <ol type="1">
           * <li>Appel API pour récupérer les GPX</li>
           * <li>Parse les GPX</li>
           * <li>Re-projection des features</li>
           * <li>Traitement des propriétés (dates, attachements...)</li>
           * <li>Mappage des features vers les layers par type de géométrie</li>
           * <li>Appel API pour importer les photos</li>
           * <li>Re-projection des features</li>
           * <li>Mappage des features vers les layers par type de géométrie</li>
           * <li>Ajoute les layers GPX à la couche d'import de KIS (Technique/Importation)</li></ol>
           */
          scope.importFile = () => {
            console.log('import');
            const val = false;
            if (scope.dropzoneComponent.files.length > 0) {

              // nettoie la console car polluée par dropzone
              // console.clear();

              // toggle variables globales
              hasGPX = false;
              scope.waitImport = true;
              processId = scope.uploadProcessID;
              importgpxservice.processId = scope.uploadProcessID;

              // import des fichiers GPX du lot glissé/déposé
              ImportExportFactory.loadFileAsString(scope.uploadProcessID).then(
                filesResponse => {
                  scope.layers = [];
                  let fileName = parseGPXtoGeoJSON(filesResponse);
                  if (fileName) {
                    parsePhotoToGeoJSONAndBuildImportLayer(fileName);
                  } else{
                    parsePhotoToGeoJSONAndBuildImportLayer(getDefaultFileName());
                  }
                },
                error => {
                  scope.waitImport = false;
                  require('toastr').error(
                    $filter('translate')('importgpxwidget.importError')
                        + error && error.detail ? error.detail : '');
                });
            }
          };

          /**
           * Transforme les fichiers GPX en geojson
           * @param filesResponse response contenant la liste de fichiers GPX sous la forme de string
           * @returns {string} retourne le nom de fichier à utiliser sur toutes les couches
           * @see importFile
           */
          function parseGPXtoGeoJSON(filesResponse) {

            let filename = null;

            for (let gpx of filesResponse.data) {

              // parse string en XML avec jQuery
              const xmlFields = $(gpx);

              if (xmlFields.length === 2) {

                // un fichier GPX a été trouvé
                hasGPX = true;

                // parse XML en geojson
                // @see https://github.com/mapbox/togeojson
                const geojson = toGeoJSON.gpx(xmlFields[1]);

                // regarde le name dans le XML au cas où la propriété n'aurait pas été convertie
                recoverPropertyWhenConversionFailed(xmlFields[1], geojson, 'name');
                recoverPropertyWhenConversionFailed(xmlFields[1], geojson, 'desc');

                // construit le préfixe des noms de couche à partir du nom de la 1ère track
                filename = defineLayernameBody(xmlFields[1]);

                // construit le suffixe des noms de couche à partir du type de géométrie de chaque objet
                for (let i = 0; i < geojson.features.length; i++) {

                  const feature = geojson.features[i];

                  // détermine le suffixe de la couche à laquelle cette feature doit appartenir
                  const suffix = defineLayernameSuffixByGeometryType(feature);

                  const layername = filename + suffix;

                  // reprojète de EPSG:4326 vers mapProjection
                  reprojectFeature(feature);

                  // classe les features en couche suivant le type de géométrie
                  manageAttachementsAndDatestring(feature);

                  addFeatureToLayer(layername, feature);
                }
                console.log('geojson pré-traité (attachments, dates) : ');
                console.log(geojson)
              }
            }
            return filename;
          }

          /**
           * Vérifie si le nom de la feature est vide.
           * Si oui, va chercher le nom dans le XML
           * @param xmlNode noeud du XML contenant toutes les tracks et tous les waypoints
           * @param geojson geojson parsé par toGeoJSON
           * @param toRecoverProperty nom de la propriété à récupérer dans le XML
           * @see parseGPXtoGeoJSON
           */
          function recoverPropertyWhenConversionFailed(xmlNode, geojson, toRecoverProperty) {
            const waypoints = xmlNode.getElementsByTagName('wpt');
            const tracks = xmlNode.getElementsByTagName('trk');
            let index = 0;
            let indexPoint = 0;
            let indexLineString = 0;
            for (const feature of geojson.features) {
              if (feature.properties[toRecoverProperty] === '') {
                let collection;
                if (feature.geometry.type === 'Point') {
                  collection = waypoints;
                  index = indexPoint;
                  indexPoint++;
                } else {
                  collection = tracks;
                  index = indexLineString;
                  indexLineString++;
                }
                // on regarde dans le XML, l'objet situé au même rang
                if (collection && collection.length > 0 && collection.item(index)) {
                  let xmlProperty = collection.item(index).getElementsByTagName(
                    toRecoverProperty).item(0).innerHTML;
                  xmlProperty = cleanCData(xmlProperty);
                  feature.properties[toRecoverProperty] = xmlProperty;

                  // récupère les éventuels attachements non convertis
                  if (!feature.properties.links && !feature.properties.attachments) {
                    const attachements = [];
                    const extensions = collection.item(index).getElementsByTagName('extensions');
                    for (const extension of extensions) {
                      const oms = extension.getElementsByTagName('om:oruxmapsextensions');
                      for (const om of oms) {
                        const exts = om.getElementsByTagName('om:ext');
                        for (const ext of exts) {
                          if (ext.getAttribute('type')==='IMAGEN'){
                            const absPath = ext.innerHTML;
                            const attachmentName = absPath.split('/').pop();
                            attachements.push(attachmentName);
                          }
                        }
                      }
                    }
                    if (attachements.length > 0){
                      feature.properties.attachments = attachements.join(',');
                    }
                  }
                }
              }
            }
          }

          /**
           * Détermine le suffixe de la couche à laquelle appartient la feature
           * @param feature objet geojson
           * @returns {string} suffixe du nom de la couche suivant le type de géométrie de la feature
           * @see parseGPXtoGeoJSON
           */
          function defineLayernameSuffixByGeometryType(feature) {
            let suffix;
            switch (feature.geometry.type) {
              case 'LineString':
                suffix = '_line';
                break;
              case 'Point':
                suffix = '_point';
                break;
              default:
                suffix = '_polygon';
            }
            return suffix;
          }

          /**
           * Vérifie que toutes les features GPX contiennent la propriété "attachments"
           * lorsque au moins 1 des features de la layer contient la propriété "attachments"
           * @see addFeatureToLayer
           */
          function propagateAttachmentPropertyInLayer(){
            for (const layer of scope.layers)
              if (layer && layer.geojson && layer.geojson.features){
                const oneFeatWithAttachment = layer.geojson.features.find(f => f.hasOwnProperty('properties') && f.properties.hasOwnProperty('attachments'));
                if (oneFeatWithAttachment){
                  for (const feature of layer.geojson.features){
                    feature.properties.attachments = feature.properties.hasOwnProperty('attachments') ? feature.properties.attachments : '';
                  }
                }
              }
          }

          /**
           * Appel Factory pour récupérer les métadonnées des photos présentes dans un sous-dossier du processus d'import<br>
           * Méthode qui lance l'exécute la construction de la couche KIS<br>
           * Convertit la date de création du fichier de timestamp à une date en string<br>
           * Convertit les DTO de type PhotoAttachmentMetadata en feature de geojson<br>
           * @param filename corps du nom de la couche (20 premiers caractères du nom de la 1ère track du GPX ou bien "gpx_yyyy-MM-dd")<br>
           * @see importFile
           */
          function parsePhotoToGeoJSONAndBuildImportLayer(filename) {
            // import des photos du lot glissé/déposé
            // si existe un sous dossier "photos" dans le dossier du processus
            ImportExportFactory.getProcessPicturesExifMetadata(scope.uploadProcessID,
              mapProjection).then(
              response => {
                const data = {
                  photos: [],
                  errors: []
                }
                if (response && response.data && Object.entries(response.data).length > 0 && response.data.photos
                      && response.data.errors) {
                  data.photos = response.data.photos;
                  data.errors = response.data.errors;
                }
                if (data.photos.length > 0) {

                  let waypointLayer;
                  const waypointLayerIndex = scope.layers.findIndex(
                    layer => layer.name.endsWith('_point'));
                  if (waypointLayerIndex >= 0){
                    waypointLayer = scope.layers[waypointLayerIndex];
                  }

                  const layername = filename + '_photo';

                  for (const photo of data.photos) {

                    // si la photo est référencée dans un waypoint alors on ne considère pas la photo comme une photo géo-taguée
                    // mais comme une photo attachée au waypoint
                    let photoIsWaypointAttachment = false;

                    if (waypointLayer){
                      for (const feature of waypointLayer.geojson.features){
                        if (feature.properties.attachments === photo.filename){
                          photoIsWaypointAttachment = true;
                        }
                      }
                    }

                    // si le nom de la photo n'est aps attachée à un waypoint alors on l'importe comme une photo géo-taguée
                    if (!photoIsWaypointAttachment){
                      // convertit les métadonnées en feature geojson
                      const feature = pictureMetadataToGeoJSON(photo);

                      // reprojète de EPSG:4326 vers mapProjection
                      reprojectFeature(feature);

                      addFeatureToLayer(layername, feature);
                    }
                  }
                }

                displayErrors(data.errors);

                if (hasGpxOrPicture(data, hasGPX)) {
                  createImportationLayer();
                }
              },
              () => {
                require('toastr').error(
                  $filter('translate')('importgpxwidget.picturesimportError'));
              }
            )
          }

          /**
           * Converti les métadonnées d'une photo en feature geojson
           * @param photo DTO de type PhotoAttachmentMetadata contenant les métadonnées d'une photo
           * @returns {feature} objet Point à inclure dans une layer GPX
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           */
          function pictureMetadataToGeoJSON(photo) {
            if (photo) {
              photo.z = photo.z === 0.0 ? 0 : photo.z;
              return {
                geometry: {
                  type: 'Point',
                  coordinates: [photo.x, photo.y, photo.z],
                },
                properties: {
                  name: photo.filename.split('.')[0],
                  time: $filter('date')(new Date(photo.datecreation),
                    FORMAT_DATE),
                  type: 'photo-tag',
                  attachments: photo.filename
                },
                type: 'Feature'
              };
            }
          }

          /**
           * Affiche les erreurs sous la forme de taustr warning
           * @param errors liste des noms de photos déposées n'ayant pas de métadonnées GPS
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           */
          function displayErrors(errors) {
            if (errors.length > 0) {
              for (const error of errors) {
                require('toastr').warning(
                  $filter('translate')('importgpxwidget.nometadata') + error);
              }
            }
          }

          /**
           * Vérifie si au moins 1 GPX ou 1 photo a été convertie
           * @param data corps de la réponse de l'appel API getProcessPicturesExifMetadata
           * @param gpxExists est true si un fichier GPX a été déposé
           * @returns {boolean} true si 1 fichier, au moins, a été converti
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           */
          function hasGpxOrPicture(data, gpxExists) {
            let hasFile = true;
            // si aucune photo et aucun GPX on cache le spinner
            if (data.photos.length === 0 && !gpxExists) {
              scope.waitImport = false;
              hasFile = false;
            }
            return hasFile;
          }

          /**
           * Remplie la couche technique d'importation KIS à partir des layers GPX (scope.layers)<br>
           * Centre la carte sur les objets de toutes les couches
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           */
          function createImportationLayer() {
            //rend actif l'onglet de la liste de couches
            scope.gpxtabs.activeTab = 1;
            scope.tabs[1].disabled = false;

            for (let i = 0; i < scope.layers.length; i++) {

              // objets d'un type
              let features = parser.readFeatures(
                scope.layers[i].geojson);

              // affecte les objets d'un type d'objet à une couche
              scope.layers[i].features = features;

              // ajoute les objets de la couche à la carte courante
              gclayers
                .getImportLayer()
                .getSource()
                .addFeatures(features);

              // applique le stype (marker différent pour les photos GPS)
              gclayers
                .getImportLayer().setStyle(myStyle);
              scope.layers[i].visible = true;
            }

            // centre la carte sur les objets
            scope.zoomAll();

            // stop spinner
            scope.waitImport = false;

            // désactive la dropzone
            scope.dragBox.setActive(false);
            scope.map.removeInteraction(scope.dragBox);
            scope.allv = true;

            for (let i = 0; i < scope.layers.length; i++) {
              console.log('geojson ' + scope.layers[i].name + ' : ');
              console.log(scope.layers[i].geojson)
            }

            // màj des layers dans le service
            importgpxservice.layers = scope.layers
          }

          /**
           * Affecte le nom de la 1ère trace aux couches du fichier
           * @param xmlNode root du document XML
           * @returns {string} partie principale du nom de couche sinon "gpx_yyy-MM-dd" quand il n'y a pas de trace
           * @see parseGPXtoGeoJSON
           */
          function defineLayernameBody(xmlNode) {
            if (xmlNode) {
              const tracks = xmlNode.getElementsByTagName('trk');

              // récupère le nom de la 1ère track
              if (tracks.length > 0 && tracks.item(
                0).getElementsByTagName('name').length > 0) {

                let trackName = tracks.item(0).getElementsByTagName(
                  'name').item(0).innerHTML;

                trackName = cleanCData(trackName);

                // tronque le nom à 20 caractères
                return trackName.length > 25 ? trackName.substring(0, 25)
                  : trackName;
              } else {
                return getDefaultFileName();
              }
            }
          }

          /**
           * Récupère la chaîne de caractères incluse dans un string préfixée par [CDATA[[
           * @param innerHtml string brut comprenant le préfixe
           * @return {string} chaîne de caractères nettoyée
           * @see defineLayernameBody
           * @see recoverPropertyWhenConversionFailed
           */
          function cleanCData(innerHtml) {
            if (innerHtml && innerHtml.includes('[CDATA[')) {
              return innerHtml.split('[CDATA[').pop().split(']]').shift();
            } else {
              return innerHtml;
            }
          }

          /**
           * Construit le nom de fichier par défaut à partir de la date du jour formatée yyyy-MM-dd
           * @returns {string} contenant "gpx_" en préfixe et la date formatée en suffixe
           * @see importFile
           * @see defineLayernameBody
           */
          function getDefaultFileName() {
            // On renvoie "gpx_YYYY-mm-DD" si la 1ère track n'a pas de nom
            const d = new Date();
            const date = [
              d.getFullYear(),
              ('0' + (d.getMonth() + 1)).slice(-2),
              ('0' + d.getDate()).slice(-2)
            ].join('-');
            return 'gpx_' + date;
          }

          /**
           * Ajoute la feature courante au geojson de la couche correspondant au type de géométrie
           * Fonction servant à réduire la taille de la méthode importFile
           * Supprime les propriétés par défaut d'un GPX inutiles pour l'import (sym, coordTimes)
           * @param feature objet courant, élément du tableau "features" d'un geojson
           * @see parseGPXtoGeoJSON
           */
          function manageAttachementsAndDatestring(feature) {

            // supprime l'attribut "sym" existant par défaut dans un gpx
            // "sym" correspond à un nom d'icône dans un logiciel d'édition gpx (Basecamp apr exemple)
            if (feature.properties.sym) {
              delete feature.properties.sym;
            }
            // supprime l'attribut "coordTimes" existant par défaut dans un gpx
            // "coordTimes" est la date de la mesure GPS du point
            if (feature.properties.coordTimes){
              delete feature.properties.coordTimes;
            }

            // créé une propriété "attachments" si l'objet GPX contient des photos ou des documents attachés
            manageAttachements(feature);

            // nettoyage de la propriété "time" pour pouvoir être convertie en date par le serveur (si on la mappe à un champ date)
            if (feature.properties.time) {
              const date = new Date(feature.properties.time);
              feature.properties.time = $filter('date')(date, FORMAT_DATE);
            }
          }

          /**
           * Ajoute le feature dans le tableau de features du geojson de la couche contenant des géométries du même type que la feature
           * @param layername nom de la couche GPX
           * @param feature objet contenant les attachments créé, auparavant, à partir d'un template
           * @see parseGPXtoGeoJSON
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           */
          function addFeatureToLayer(layername, feature) {
            const geomLayerIndex = scope.layers.findIndex(
              layer => layer.name === layername);
            if (scope.layers.length !== 0 && geomLayerIndex
                >= 0) {
              // la couche existe déjà
              scope.layers[geomLayerIndex].geojson.features.push(
                feature);
              scope.layers[geomLayerIndex].geojson.totalFeatures++;
            } else {
              // la couche n'existe pas encore
              const geojson = {
                type: 'FeatureCollection',
                totalFeatures: 1,
                features: [feature],
                crs: {}
              }
              scope.layers.push({
                name: layername,
                geojson: geojson
              })
            }
            // ajoute la propriété "attachments à toutes les features d'une layer
            // lorsqu'il existe au moins 1 feature ayant des fichiers attachés
            propagateAttachmentPropertyInLayer();
          }

          /**
           * Nettoie les noms des fichiers attachés de l'objet GPX et insère ces noms dans une propriété "attachements" (tableau)
           * les fichiers attachés sont écrit sous la forme d'un chemin absolu dont on récupère uniquement le nom du fichier + extension
           * @param feature objet de la couche GPX
           * @see manageAttachementsAndDatestring
           */
          function manageAttachements(feature) {
            const links = [];

            // traitement uniquement si l'objet a une propriété "links" pour contenir des fichiers attachés
            if (feature.properties.links) {
              for (const link of feature.properties.links) {
                for (let absolutePath of Object.values(link)) {

                  // les photos sont écrites avec un chemin absolu dont on récupère uniquement le nom du fichier + extension
                  // limite la recherche aux extensions autorisées
                  if (allowedAttachments.test('' + absolutePath)) {
                    if (absolutePath.length > 4 && absolutePath.includes('.')) {

                      let urlEncodedFilename = null;

                      // récupération du nom de fichier avec prise en compte du séparateur sous windows
                      // et avec prise en compte de l'encodage URL (ex. "\" encodé devient "%2F")
                      if (absolutePath.includes('\\')) {
                        // si windows
                        const pathSeparByASlash = absolutePath.split('\\');
                        urlEncodedFilename = pathSeparByASlash[pathSeparByASlash.length - 1];

                      } else if (absolutePath.includes('%2F')) {
                        // si windows et URL-encoded
                        const pathSeparByUnicode = absolutePath.split('%2F');
                        urlEncodedFilename = pathSeparByUnicode[pathSeparByUnicode.length - 1];

                      } else if (absolutePath.includes('/')) {
                        // si unix
                        const pathSeparBySlash = absolutePath.split('/');
                        urlEncodedFilename = pathSeparBySlash[pathSeparBySlash.length - 1];
                      }
                      if (urlEncodedFilename) {
                        // URL-decode (ex. "%20" décodé devient " ")
                        const filename = urlEncodedFilename.replace('%20', ' ').replace('%C3%A9',
                          'é').replace('%C3%A8', 'è').replace('%C3%A7', 'ç').replace('%26',
                          '&').replace('%7C', '|').replace('%C3%B4', 'ô').replace(
                          '%C3%A0', 'à').replace('%C3%AA', 'ê');
                        links.push(filename);
                      }
                    }
                  } else {
                    const extensionOrPath = absolutePath.includes('.')
                      ? absolutePath.split('.')[absolutePath.split('.').length
                        - 1].toUpperCase() : absolutePath;
                    require('toastr').warning(
                      $filter('translate')('importdxfwidget.format')
                        + extensionOrPath + $filter('translate')(
                        'importdxfwidget.formatnotallowed'));
                  }
                }
              }
            }
            // supprime la propriété links dans tous les cas
            delete feature.properties.links;

            // définie une nouvelle propriété "attachments" à partir de la variable "links"
            if (links.length > 0) {
              if (feature.properties.attachments && feature.properties.attachments.length > 0){
                feature.properties.attachments = ',' + links.join(',');
              }else{
                feature.properties.attachments = links.join(',');
              }
            }
          }

          /**
           * Reprojète objet par objet depuis EPSG:4326 vers la projection de la carte
           * @param feature objet courant issu d'une des couches de la carte (scope.layers)
           * @see parsePhotoToGeoJSONAndBuildImportLayer
           * @see parseGPXtoGeoJSON
           */
          function reprojectFeature(feature) {
            if (feature && feature.geometry && feature.geometry.coordinates) {
              if (feature.geometry.type === 'Point') {
                feature.geometry.coordinates = ol.proj.transform(
                  feature.geometry.coordinates,
                  'EPSG:4326', mapProjection);
              } else {
                const coordinates = [];
                for (let i = 0; i < feature.geometry.coordinates.length; i++) {
                  coordinates.push(
                    ol.proj.transform(feature.geometry.coordinates[i],
                      'EPSG:4326', mapProjection));
                }
                feature.geometry.coordinates = coordinates;
              }
            }
          }

          /**
           * définition de la sélection par cadre
           */
          scope.dragBox = new ol.interaction.DragBox({
            condition: function (evt) {
              //MacEnvironments don't get here because the event is not
              //recognized as mouseEvent on Mac by the google closure.
              //We have to use the apple key on those devices
              return (
                evt.originalEvent.ctrlKey || scope.selectionisactive
              ); /* ||
                              (gaBrowserSniffer.mac && evt.originalEvent.metaKey);*/
            },
            style: gcStyleFactory.getStyle('selectrectangle'),
          });

          scope.dragBox.set('gctype', 'kis');
          scope.dragBox.set('interaction', 'Select');
          scope.dragBox.set('widget', 'ImportGPX');

          scope.dragBox.setActive(false);

          /**
           * création de la sélection après sélection par cadre
           */
          scope.dragBox.on('boxend', function () {
            scope.selectedfeatures = [];
            let i = 0;
            gclayers
              .getImportLayer()
              .getSource()
              .forEachFeatureIntersectingExtent(
                scope.dragBox.getGeometry().getExtent(),
                function (feature) {
                  feature.set('index', ++i);
                  scope.selectedfeatures.push(feature);
                }
              );
            SelectManager.clear();
            try {
              SelectManager.addFeaturesFromGeojson(
                JSON.parse(parser.writeFeatures(scope.selectedfeatures))
              );
            } catch (e) {
              console.info("features n'ont pas d'IDs normaux");
            }
            boutonDroit();
          });

          /**
           * Sélectionne des objets, résultat de la recherche par rectangle
           */
          scope.selectElementOnMap = () => {
            scope.selectionisactive = !scope.selectionisactive;
            if (!scope.selectionisactive) {
              scope.selectedfeatures = [];
              SelectManager.clear();
              scope.dragBox.setActive(false);
              scope.map.removeInteraction(scope.dragBox);
            } else {
              scope.dragBox.setActive(true);
              gcInteractions.setCurrentToolBar(scope.toolBarWidget);
              scope.map.addInteraction(scope.dragBox);
            }
          };

          /**
           * Gestion des fonction Zoom +/- du bouton droit
           */
          function boutonDroit() {
            const Supprimer = [
              $filter('translate')('importdxfwidget.remove'),
              function () {
                scope.removeFeatures();
              },
            ];
            const clear = [
              $filter('translate')('importdxfwidget.abandon'),
              function () {
                clearFeatures();
              },
            ];
            if (scope.menuContext) {
              let menu = scope.menuContext;
              if (scope.menuContext.length >= 3) {
                scope.menuContext.splice(0, scope.menuContext.length - 2);
              }
              menu.unshift(clear);
              menu.unshift(Supprimer);
            }
          }

          /**
           * Supprime des objets de la couche d'importation
           */
          scope.removeFeatures = () => {
            if (scope.selectedfeatures.length > 0) {
              const ans = window.confirm(
                $filter('translate')('common.confirm_action')
              );

              if (ans) {

                // copie la couche d'importation KIS dans une collection
                const olCollection = new ol.Collection(
                  gclayers
                    .getImportLayer()
                    .getSource()
                    .getFeatures()
                );

                // enlève les objets sélectionnées de la collection
                for (const ft of scope.selectedfeatures) {
                  olCollection.remove(ft);
                }
                // vide la couche d'importation KIS
                gclayers
                  .getImportLayer()
                  .getSource()
                  .clear();

                // insère la collection dans la couche d'importation KIS
                gclayers
                  .getImportLayer()
                  .getSource()
                  .addFeatures(olCollection.getArray());

                let ids = [];

                for (let i = 0; i < scope.layers.length; i++) {

                  // copie les objets de la couche GPX dans une collection
                  const layer = scope.layers[i];
                  const col = new ol.Collection(layer.features);

                  // supprime les objets sélectionnés de la collection
                  for (const ft of scope.selectedfeatures) {
                    col.remove(ft);
                  }
                  // remplace les objets de la couche GPX par la collection
                  layer.features = col.getArray();
                  layer.geojson = JSON.parse(
                    parser.writeFeatures(layer.features)
                  );
                  layer.geojson.totalFeatures = layer.geojson.features.length;
                  if (layer.geojson.totalFeatures === 0) {
                    ids.push(i);
                  }
                }
                if (ids.length > 0) {
                  ids = ids.reverse();
                  ids.map((x) => {
                    return scope.layers.splice(x, 1);
                  });
                }
                // supprime les données sélectionnées sans désactiver le bouton de sélection
                clearFeatures();
                scope.selectElementOnMap();
                // màj des layers dans le service
                importgpxservice.layers = scope.layers;
              }
            } else {
              require('toastr').info(
                $filter('translate')('importdxfwidget.selectObjectfirst')
              );
            }
          };

          /**
           * Vide la sélection de features
           * Supprime l'intéraction de boîte sur la carte
           */
          function clearFeatures() {
            const ids = [];
            if (scope.menuContext) {
              for (let i = 0; i < scope.menuContext.length; i++) {
                if (
                  scope.menuContext[i][0] ===
                    $filter('translate')('importdxfwidget.remove') ||
                    scope.menuContext[i][0] ===
                    $filter('translate')('importdxfwidget.abandon')
                ) {
                  ids.push(i);
                }
              }
            }
            ids.reverse().map((x) => {
              return scope.menuContext.splice(x, 1);
            });
            scope.selectedfeatures = [];
            SelectManager.clear();
            scope.selectionisactive = false;
            scope.dragBox.setActive(false);
            scope.map.removeInteraction(scope.dragBox);
          }

          /**
           * Ouvre la table de correspondance des attributs
           */
          scope.listFeatures = () => {
            if (SelectManager.getpop()) {
              try {
                if (SelectManager.getpop().element) {
                  SelectManager.getpop().destroy();
                }
                if (SelectManager.getpop().scope) {
                  SelectManager.getpop().scope.$broadcast('$destroy');
                }
              } catch (e) {
                SelectManager.setpop(null);
              }
            }
            pop = gcPopup.open({
              scope: scope,
              title:
                  $filter('translate')('importdxfwidget.features') +
                  ` <label class="label label-default gpxSelectedFeaturesLength">${scope.selectedfeatures.length}</label>`,
              template:
                  'js/XG/widgets/mapapp/importdxf/views/importdxffeatures.html',
              showClose: true,
            });
            SelectManager.setpop(pop);
          };

          // ajoute la mise en évidence colorée orange
          scope.highLightFeature = (f) => {
            gclayers.addhighLightFeature(f);
          };
          // supprime la mise en évidence colorée orange
          scope.removehighLightFeature = (f) => {
            $('.gpxSelectedFeaturesLength').text(''+ scope.selectedfeatures.length);
            gclayers.removehighLightFeatures(f);
          };

          /**
           * Zoom sur une feature d'une couche du GPX
           * Attention! Il faut modifier l'extent car fit() sur un extent de point plante (error 32)
           * @param feature feature dont on récupère l'extent
           */
          scope.zoomOnFeature = (feature) => {
            const extent = feature.getGeometry().getExtent();
            if (isExtentAPoint(extent)) {
              fitMapToPoint(extent);
            } else {
              scope.map
                .getView()
                .fit(extent, scope.map.getSize());
            }
          };

          /**
           * Enlève un objet de la sélection
           * @param ids liste d'id
           */
          scope.removeFeature = (ids) => {
            scope.selectedfeatures.splice(ids, 1);
            SelectManager.clear();
            try {
              SelectManager.addFeaturesFromGeojson(
                JSON.parse(parser.writeFeatures(scope.selectedfeatures))
              );
            } catch (e) {
              console.info("features n'ont pas d'IDs normal");
            }
          };

          /**
           * Zoom global pour voir tous les objets du GPX à l'écran
           * Adapte la vue si l'extent est un point
           * @see createImportationLayer
           * @see resetViewAfterNavigation
           */
          scope.zoomAll = () => {
            const extent = gclayers
              .getImportLayer()
              .getSource()
              .getExtent();
            if (isExtentAPoint(extent)) {
              fitMapToPoint(extent);
            } else {
              scope.map.getView().fit(extent, scope.map.getSize());
            }
          };

          /**
           * Supprime les fichiers déposés
           * Vide la couche d'importation KIS
           * Supprime les fichiers dans le dossier du processus d'import du repo
           */
          scope.clearDropzone = () => {
            if (scope.dropzoneComponent.files.length > 0) {
              scope.dropzoneComponent.removeAllFiles();
            }
            gclayers
              .getImportLayer()
              .getSource()
              .clear();
            scope.layers = [];
            importgpxservice.layers = [];
            scope.waitImport = false;
            scope.tabs[1].disabled = true;
            ImportExportFactory.removeProcessFiles(scope.uploadProcessID).then(
              () => {
              },
              (error) => {
                require('toastr').error(
                  $filter('translate')('importgpxwidget.removeallerror') + error.detail);
              }
            )
          };

          /**
           * Gère l'état commun de la visibilité de chaque couche
           * Si false, vide la couche d'importation KIS
           * Si true, ajoute les layers GPX à la couche d'importation KIS
           */
          scope.toggleAllvisible = () => {
            scope.allv = !scope.allv;
            for (const layer of scope.layers) {
              layer.visible = scope.allv;
            }
            gclayers
              .getImportLayer()
              .getSource()
              .clear();
            if (scope.allv) {
              for (const layer of scope.layers) {
                gclayers
                  .getImportLayer()
                  .getSource()
                  .addFeatures(layer.features);
              }
            }
          };

          /**
           * Gère la visibilité d'un seul objet
           * @param index ralo de l'objet dans la liste de couches
           */
          scope.toggleVisible = (index) => {

            // vide la couche d'importation KIS
            gclayers
              .getImportLayer()
              .getSource()
              .clear();

            // toggle la variable visible de la layer au rang index
            if (scope.layers[index]) {
              scope.layers[index].visible = !scope.layers[index].visible;
            }
            // ajoute les couches GPX qui ont la propriété visible égale à true à la couche d'importation KIS
            let allvisiblelayers = 0;
            for (let i = 0; i < scope.layers.length; i++) {
              if (scope.layers[i].visible) {
                gclayers
                  .getImportLayer()
                  .getSource()
                  .addFeatures(scope.layers[i].features);
                scope.layers[i].visible = true;
                allvisiblelayers++;
              }
            }

            // si toutes les couches sont masquées unitairement on désactive le bouton global
            if (allvisiblelayers === scope.layers.length) {
              scope.allv = true;
            } else if (allvisiblelayers === 0) {
              scope.allv = false;
            }
          };

          /**
           * Action au clic sur le bouton "Copier dans une couche"
           * Ouvre la popup de sélection de composant et mappage des champs
           * @param layer couche GPX
           * @see gcimportfeatures
           */
          scope.copyIn = (layer) => {
            if (dialog) {
              dialog.close();
            }
            scope.selectedLayer = layer;
            if (!scope.selectedLayer.importcfg) {
              scope.selectedLayer.importcfg = {
                liaisons: {},
                values: {},
              };
            }
            scope.layerAttributes = Object.keys(
              layer.geojson.features[0].properties
            );
            dialog = extendedNgDialog.open({
              template:
                  'js/XG/widgets/mapapp/importdxf/views/importdxftemplate.html',
              className:
                  'ngdialog-theme-plain overflowY width750 max-height500 miniclose',
              closeByDocument: false,
              scope: scope,
              title:
                  $filter('translate')('importdxfwidget.copytitle')
                  + layer.name,
              draggable: true,
            });
          };

          /**
           * Récupère les attributs mappés de type g2c.attachment ou g2c.attachments
           * @param ftiuid uid du fti cible
           * @param geojson geojson de la couche en cours de sauvegarde
           * @return {string[]} tableau des noms d'attributs attachment mappés dans le fti cible
           */
          function findAttachmentAttributes(ftiuid, geojson) {
            const attributes = []
            if (geojson.features.length > 0) {
              const fti = FeatureTypeFactory.getFeatureByUid(ftiuid);
              const properties = Object.keys(geojson.features[0].properties);
              for (const key of properties) {
                const attribute = fti.attributes.find(
                  attr => attr.name === key && attr.type === 'g2c.attachment' || attr.type
                        === 'g2c.attachments');
                if (attribute) {
                  attributes.push(attribute.name);
                }
              }
            }
            return attributes;
          }

          /**
           * Lance l'enregistrement des objets dans le fti choisi
           * Fonction transmise au composant gcimportfeatures
           * @see gcimportfeatures
           */
          scope.save = () => {
            $timeout(() => {
              addFeaturesToFtid(scope.selectedLayer);
              dialog.close();
            });
          };

          /**
           * Effectue l'appel Factory pour mettre à jour le fti après ajout des nouveaux objets
           * @param layer couche ne contenant qu'un seul type de géométrie
           * @see scope.save
           */
          function addFeaturesToFtid(layer) {
            gaDomUtils.showGlobalLoader();
            const geojson = layer.geojson;
            defaultLayer = angular.copy(geojson);

            // parse en timestamp les propriété source mappées vers un champ cible de type timestamp
            parseDateStringPropertiesToTimestamp(geojson, layer.importcfg);

            // remplace les noms de propriétés mappées par les noms des attributs du fti cible
            // supprime les autres propriétés
            //transformGeojsonProperties(layer);

            for (const feature of geojson.features) {

              // redéfinition des propriétés du geojson de chaque couche
              // pour avoir le format requis pour l'appel API:
              // avant: geojson_var_name -> fti.attribute.name
              // après: fti.attribute.name -> geojsonfeature_var_value

              let propnew = {};

              const values = Object.keys(layer.importcfg.values);
              const liaisons = Object.keys(layer.importcfg.liaisons);

              if (liaisons.length === 0 && values.length === 0) {
                propnew = {};
              } else if (liaisons.length > 0 && values.length === 0) {
                for (const [key, value] of Object.entries(layer.importcfg.liaisons)) {
                  propnew[value] = feature.properties[key];
                }
              } else if (liaisons.length === 0 && values.length > 0) {
                propnew = layer.importcfg.values;
              } else {
                propnew = layer.importcfg.values;
                for (const [key, value] of Object.entries(layer.importcfg.liaisons)) {
                  if (Object.keys(propnew).indexOf(key) === -1) {
                    propnew[value] = feature.properties[key];
                  }
                }
              }
              feature.properties = propnew;
            }


            console.log('couche en cours de sauvegarde : ');
            console.log(geojson);

            EditFactory.add(
              layer.importcfg.ftid,
              geojson,
              mapProjection,
              'false'
            ).then(
              (res) => {
                if (
                  res.data.errors.length === 0 ||
                        res.data.create.length === layer.geojson.features.length
                ) {
                  // Récupère les attributs mappés de type g2c.attachment ou g2c.attachments
                  const attributes = findAttachmentAttributes(layer.importcfg.ftid, geojson);
                  if (attributes.length > 0) {
                    FeatureAttachmentFactory.attachImportedFiles(processId,
                      layer.importcfg.ftid, attributes, res.data.create).then(
                      copy => {
                        if (copy && copy.data) {
                          if (copy.data.errors && copy.data.errors.length > 0) {
                            for (const fileNotCopied of copy.data.errors) {
                              require('toastr').warn(
                                $filter('translate')('importgpxwidget.copyfileerror') +
                                        fileNotCopied
                              );
                            }
                          }
                        }
                        resetMapAfterSave(layer.name);
                      },
                      () => {
                        require('toastr').error(
                          $filter('translate')('importgpxwidget.copyallerror'));
                      }
                    )
                  }else{
                    resetMapAfterSave(layer.name);
                  }
                } else {
                  // si erreur de sauvegarde, restauration de la layer du GPX initiale
                  layer.geojson = angular.copy(defaultLayer);
                  gaDomUtils.hideGlobalLoader();
                  require('toastr').error(
                    $filter('translate')('importdxfwidget.copyerror') +
                          layer.name
                  );
                }
              },
              (error) => {
                gaDomUtils.hideGlobalLoader();
                if (
                  error.data &&
                        error.data.code === 403 &&
                        error.data.CLASSTYPE === 'Error' &&
                        error.data.message &&
                        error.data.details
                ) {
                  gcRestrictionProvider.showDetailsErrorMessage(error);
                } else {
                  require('toastr').error(
                    $filter('translate')('importdxfwidget.copyerror') +
                          layer.name
                  );
                }
              }
            );
          }

          /**
           * Restaure la map après sauvegarde
           * Supprime la layer importée
           * Supprime la visibilité de la layer suppromée
           */
          function resetMapAfterSave(layername){
            for (let i = 0; i < scope.layers.length; i++) {

              // supprime la layer GPX
              if (scope.layers[i].name === layername) {
                scope.layers.splice(i, 1);
                break;
              }

              // si toutes les layers ont été importées alors on bascule sur l'onglet "Importer"
              if (scope.layers.length === 0) {
                scope.gpxtabs.disabled = [false, true];
                scope.gpxtabs.activeTab = 0;
              }
            }

            // enlève la layer importée de la couche Import de KIS
            scope.toggleVisible();

            // recharche l'affiche de la carte
            $rootScope.$broadcast(
              'gcOperationalLayerChange',
              '',
              'applyall'
            );

            gaDomUtils.hideGlobalLoader();
            require('toastr').success(
              $filter('translate')('importdxfwidget.copysuccess') +
                layername
            );
          }

          /**
           * Récupère la liste des calques du GPX après navigation dans les autres catégories de widgets
           */
          function resetViewAfterNavigation() {
            if (importgpxservice.layers && importgpxservice.processId){
              processId = importgpxservice.processId;
              scope.layers = importgpxservice.layers;
              scope.gpxtabs = {
                activeTab: 1,
              }
              scope.tabs[1].disabled = false;
              const extent = gclayers
                .getImportLayer()
                .getSource()
                .getExtent();
              if (isExtentAPoint(extent)) {
                fitMapToPoint(extent);
              } else {
                scope.map.getView().fit(extent, scope.map.getSize());
              }
            }
          }

          /**
           * Vérifie qu'une propriété soit mappée vers un champ timestamp
           * Si oui, transforme la date en timestamp
           * @param geojson geojson de la couche GPX contenant les features
           * @param importcfg objet contenant les liaisons (nom_propriété_geojson -> nom_champ)
           * @see addFeaturesToFtid
           */
          function parseDateStringPropertiesToTimestamp(geojson, importcfg) {

            const fti = FeatureTypeFactory.getFeatureByUid(importcfg.ftid);

            if (fti && fti.attributes) {

              // boucle sur les attributs du fti
              for (const [geojsonPropertyName, dbAttributeName] of
                Object.entries(importcfg.liaisons)) {

                // récupère l'attribut du fti
                const dbAttribute = fti.attributes.find(
                  attr => attr.name === dbAttributeName);
                if (dbAttribute && dbAttribute.name === dbAttributeName
                    && dbAttribute.type === 'java.sql.Timestamp') {

                  // si le type de l'attribut cible est "Timestamp" alors on convertit la propriété en timestamp
                  for (const feature of geojson.features) {
                    const date = feature.properties[geojsonPropertyName];
                    if (!isNaN(Date.parse(date))) {
                      feature.properties[geojsonPropertyName] = Date.parse(
                        date);
                    } else {
                      // Date.parse renvoie NaN si date invalide
                      console.log('parseDateStringPropertiesToTimestamp : \n'
                          + 'impossible de convertir la string '
                          + feature.properties[geojsonPropertyName] + '=' + date
                          + ' en timestamp');
                      require('toastr').error(
                        $filter('translate')('importgpxwidget.errorparsedate')
                          + 'feature.properties.' + geojsonPropertyName + '='
                          + date);
                    }
                  }
                }
              }
            }
          }

          /**
           * Centre la carte sur un point
           * Utile dans le cas où l'extent est un point (layer ne contenant qu'un seul point)
           * @param extent étendue courante de la carte (Xmin,Ymin,Xmax,Ymax)
           * @see zoomAll
           * @see zoomOnObject
           */
          const fitMapToPoint = (extent) => {
            if (Array.isArray(extent) && extent.length === 4) {
              const pointExtent = [extent[0] - (DISTANCE / 2),
                extent[1] - (DISTANCE / 2), extent[2] + (DISTANCE / 2),
                extent[3] + (DISTANCE / 2)];
              scope.map.getView().fit(pointExtent, scope.map.getSize());
            }
          };

          /**
           * Vérifie si extent est un point
           * @param extent étendue courante de la carte (Xmin, Ymin, Xmax, Ymax)
           * @returns {boolean} true si l'extent est un point
           */
          const isExtentAPoint = (extent) => {
            if (Array.isArray(extent) && extent.length === 4) {
              return extent[0] === extent[2] && extent[1] === extent[3];
            }
          };

          /**
           * Vérifie si le système d'exploitation est windows
           * @return {boolean} si le userAgent contient 'windows'
           */
          const isWindowsOS = () => {
            const {userAgent} = navigator;
            return userAgent.includes('Windows');
          };
        },
      };
    }
  }

  importgpxwidget.$inject = [
    'importgpxservice',
    'ImportExportFactory',
    'FeatureTypeFactory',
    'FeatureAttachmentFactory',
    'gclayers',
    '$filter',
    'extendedNgDialog',
    '$timeout',
    'EditFactory',
    '$rootScope',
    'gaDomUtils',
    'gcStyleFactory',
    'SelectManager',
    'gcPopup',
    'gcRestrictionProvider',
    'gcInteractions'
  ];
  return importgpxwidget;
});