'use strict';
define(function() {
  // Constants
  const CSS_CLASSES = {
    DIALOG: 'ngdialog-theme-plain overflowY nopadding miniclose',
    DIALOG_SMALL: 'ngdialog-theme-plain overflowY width400 nopadding miniclose',
    DIALOG_MEDIUM: 'ngdialog-theme-plain overflowY width600 miniclose',
    DIALOG_LARGE: 'ngdialog-theme-plain overflowY width1000 nopadding miniclose'
  };

  const TEMPLATES = {
    DATA_TABLE: 'js/XG/widgets/mapapp/layerManager/views/popoverDatatable.html',
    EXPRESSION_UPDATE: 'js/XG/widgets/mapapp/layerManager/views/popoverExpressionUpdate.html',
    EXPRESSION_FILTER: 'js/XG/widgets/mapapp/layerManager/views/popoverExpressionFilter.html',
    CHARTS: 'js/XG/widgets/mapapp/layerManager/views/charts/charts.html'
  };
  /**
   * @ngdoc directive
   * @name layerManagerDataTable
   * @description
   *   Directive pour la datatable d'une couche de la carte.
   *   La directive est générée pour la couche depuis le popup menu.
   *   On injecte les dépendances suivantes :
   *   - $rootScope : le scope global
   *   - gclayers : le service des couches de la carte
   *   - extendedNgDialog : le service de gestion des dialogues
   *   - EditFactory : le service de gestion des requetes de modification
   *   - $filter : le service de filtres
   *   - SelectManager : le service de gestion des sélections
   *   - gaDomUtils : le service de gestion des éléments du DOM
   *   - ActionsManager : le service de gestion des actions
   *
   *   @param {object} $rootScope
   *   @param {object} gclayers
   *   @param {object} extendedNgDialog
   *   @param {object} EditFactory
   *   @param {object} $filter
   *   @param {object} SelectManager
   *   @param {object} gaDomUtils
   *   @param {object} ActionsManager
   *
   * @return {object} La directive
   */
  const layerManagerDataTable = function(
    $rootScope,
    gclayers,
    extendedNgDialog,
    EditFactory,
    $filter,
    SelectManager,
    gaDomUtils,
    ActionsManager,
    QueryFactory
  ) {
    return {
      /** @property {String} templateUrl Url of the directive UI */
      templateUrl:
        'js/XG/widgets/mapapp/layerManager/views/layerManagerDataTable.html',
      /** @type {String} [description] */
      restrict: 'AE',

      /**
       * link
       *
       * @param scope
       * @param element
       * @param attrs
       * @param ctrl
       */
      link: function(scope) {


        /**
         * Initialise le filtre de la table de données selon le filtre CQL de la couche.
         * Si la couche a un filtre CQL par défaut (différent de '1=1'), applique ce filtre
         * à la table et l'active.
         *
         * @private
         */
        const initialize = () => {
          // applique le filtre existant de la couche dans la datatable
          if (!scope.fromActiveFilter && scope.layer
            && scope.layer.cql_filter && scope.layer.cql_filter !== '1=1'){
            if (!scope.filtertemp) {
              scope.filtertemp = {};
            }
            scope.filtertemp.filter = scope.layer.cql_filter;
            scope.initialFilterOfLayer = scope.layer.cql_filter;
          }
        };



        /**
         * Au clic sur le bouton suppimer (croix) de la zone de saisie du filtre des couches
         */
        scope.resetFilter = () => {
          scope.filtertemp.filter = '';
        };


        scope.closepanel = () => $rootScope.panelsManager.removePanel('donnees');

        scope.activefilter = (f) => {
          if ($rootScope.xgos && $rootScope.xgos.sector === 'hpo' && $rootScope.xgos.hpo) {
            $rootScope.xgos.hpo.clauseWhere = f;
            if (scope.map && scope.map.getView) {
              $rootScope.xgos.hpo.srid = scope.map.getView().getProjection().getCode();
            }
          }
          scope.wheretab = f;
          scope.closepanel();
          const panelScope = $rootScope.$new();
          panelScope.map = scope.map;
          panelScope.roles = scope.roles;
          panelScope.coordsshow = true;
          panelScope.layer = scope.layer;
          panelScope.currlayer = scope.layer;
          panelScope.currentfti = scope.layer.get('fti');
          panelScope.ftiuid = panelScope.currentfti.uid;
          panelScope.wheretab = f;
          panelScope.filtertemp = { filter: f };
          panelScope.currentselectedActionOnComponentTable = {
            value: {},
          };              // the default height of the datatable
          panelScope.layerdatatable = {};
          panelScope.layerdatatable.height = 320;
          panelScope.fromActiveFilter = true;
          $rootScope.panelsManager.addPanel({
            id: 'donnees',
            stickToRight: true,
            templateUrl: TEMPLATES.DATA_TABLE,
            scope: panelScope,
            stickToBorder: true,
            visible: true,
            resizable: true,
          });
        };


        scope.tableResult = {
          elements: {
            features: [],
          },
        };

        scope.executeActionOnComponentTable = () => {
          if (scope.tableResult.elements.features.length === 0) {
            require('toastr').error(
              $filter('translate')('expression_calculator.nofeatureselected')
            );
          } else {
            ActionsManager.executeActionOnComponent(
              scope.currentselectedActionOnComponentTable.value,
              scope.actionsComponent,
              scope.currentfti,
              scope.tableResult.elements,
              scope.tableResult.elements,
              []
            );
          }
        };


        scope.openExpressionUpdate = () => {
          extendedNgDialog.open({
            template: TEMPLATES.EXPRESSION_UPDATE,
            className: CSS_CLASSES.DIALOG_SMALL,
            closeByDocument: false,
            scope: scope,
            title: $filter('translate')('expression_calculator.title'),
            draggable: true,
          });
        };


        /**
         * [OpenDataTabale description]
         *
         * @param {[type]}
         *            layer [description]
         */
        scope.openExpressionFilter = () => {
          extendedNgDialog.open({
            template: TEMPLATES.EXPRESSION_FILTER,
            className: CSS_CLASSES.DIALOG_MEDIUM,
            closeByDocument: false,
            scope: scope,
            title: $filter('translate')('expression_calculator.title'),
            draggable: true,
            resizable: true
          });
        };


        // //CHART

        scope.openCharts = () => {
          extendedNgDialog.open({
            template: TEMPLATES.CHARTS,
            className: CSS_CLASSES.DIALOG_LARGE,
            closeByDocument: false,
            scope: scope,
            title: $filter('translate')('layermanager.charts'),
            draggable: true,
          });
        };
        // / END CHART

        /**
         *
         * Enregistrement de la mapmodel et map par utilisateur en mode
         * non config
         *
         */

        scope.currstyle = {};

        /**
         * [removeLayerFromMap description]
         *
         * @param {olLayer}
         *            layer
         * @return nothing
         */
        scope.editmode = false;
        scope.openeditmode = () => {
          if (scope.editmode) {
            scope.editmode = false;
          } else {
            scope.editmode = true;
          }
        };

        /**
         * Affiche une popup de confirmation avant mise à jour d'un attribut de la table en masse
         * @param filter filtre de saisie
         * @param expression valeur affectée à l'attribut
         * @param fid uid du composant
         * @param attribute name de l'attribut à modifier sur toute la table
         * @see submitUpdateAttribute
         */
        scope.updateAttribute = (filter, expression, fid, attribute) => {
          console.log(filter);
          console.log(attribute);
          if (attribute){
            swal(
              {
                title: $filter('translate')('layermanager.updateAttribute.title'),
                text: $filter('translate')('layermanager.updateAttribute.text1')
                  + attribute + $filter('translate')('layermanager.updateAttribute.text2'),
                type: 'warning',
                showCancelButton: true,
                confirmButtonColor: 'rgb(92, 184, 92)',
                confirmButtonText: $filter('translate')('common.ok'),
                cancelButtonText: $filter('translate')('common.cancel'),
                closeOnConfirm: true,
              },
              (isConfirm) => {
                if (isConfirm) {
                  scope.submitUpdateAttribute(filter, expression, fid, attribute);
                }
              });
          }else{
            require('toastr').error(
              $filter('translate')('layermanager.no_attribute_layer')
            );
          }
        };


        /**
         * Mise à jour d'un attribut de la table en masse
         * @param filter filtre de saisie
         * @param expression valeur affectée à l'attribut
         * @param fid uid du composant
         * @param attribute name de l'attribut à modifier sur toute la table
         */
        scope.submitUpdateAttribute = (filter, expression, fid, attribute) => {
          if (attribute) {
            if (filter===undefined || filter === '') {
              filter = '1=1';
            }
            const sdata = {
              attribute: attribute,
              where: filter,
              expression: expression,
            };
            gaDomUtils.showGlobalLoader();
            EditFactory.updateexpression(sdata, fid).then(
              () => {
                scope.closepanel();
                $rootScope.panelsManager.addPanel({
                  id: 'donnees',
                  stickToRight: true,
                  templateUrl: TEMPLATES.DATA_TABLE,
                  scope: scope,
                  stickToBorder: true,
                  visible: true,
                });
                gaDomUtils.hideGlobalLoader();
              },
              () => {
                gaDomUtils.hideGlobalLoader();
              }
            );
          } else {
            require('toastr').error(
              $filter('translate')('layermanager.no_attribute_layer')
            );
          }
        };

        /**
         * Met à jour le filtre de la couche courante avec la valeur de 'f'
         * et active le filtre sur la carte.
         * Si le secteur est HPO, met à jour également la clause WHERE
         * de l'objet xgos.hpo.
         *
         * @param {string} f - la valeur du filtre
         */
        scope.activeMapfilter = (f) => {
          if (
            $rootScope.xgos &&
            $rootScope.xgos.sector === 'hpo' &&
            $rootScope.xgos.hpo
          ) {
            $rootScope.xgos.hpo.clauseWhere = f;
            if (scope.map && scope.map.getView)
              $rootScope.xgos.hpo.srid = scope.map
                .getView()
                .getProjection()
                .getCode();
          }

          scope.currlayer.cql_filter = f;
          $rootScope.$broadcast(
            'gcOperationalLayerChange',
            scope.currlayer,
            'cql_filter'
          );
        };


        //KIS-3661:Supprimer la surbrillance des features sélectionnées sans impacter KIS-3625.
        //--KIS-3672: Rétablir le filtre de la couche courante
        scope.clearHighlightedFeatures = () => {
          gclayers.clearhighLightFeatures();
          SelectManager.clear();
          scope.filtertemp.filter = '';
          if (scope.initialFilterOfLayer !== scope.currlayer.cql_filter) {
            scope.currlayer.cql_filter = scope.initialFilterOfLayer;
            $rootScope.$broadcast(
              'gcOperationalLayerChange',
              scope.currlayer,
              'cql_filter'
            );
          }
        };


        const initFromRole = () => {
          scope.actionsComponent = [];
          scope.editAttributesGrantedSet = new Set();
          if ($rootScope.xgos.isroot){
            // si root, on donne les droits sur toutes les données
            scope.editFtiGranted = true;
            scope.editAttributesGranted = scope.currentfti.attributes.map(attr => attr.name);
          } else{
            if (scope.roles  && scope.roles.length > 0){
              // le rôle rootUser  donne les droits sur toutes les données
              //  (widget Edition métier notamment)
              // sans regarder l'éventuel paramétrage des couches pour ce rôle
              if (scope.roles && scope.roles.findIndex(f => f.name === 'rootUser') > -1){
                scope.editFtiGranted = true;
                scope.editAttributesGranted
                  = scope.currentfti.attributes.map(attr => attr.name);
              }else{
                for (const role of scope.roles){
                  // index du fti dans les droits de l'utilisateur
                  const ftiAuthorizationIndex
                    = role.authorizations.findIndex(ft => ft.item === scope.ftiuid);

                  // vérifie que le role donne un droit d'écriture sur le composant
                  scope.currentRoleAllowEditFti = ftiAuthorizationIndex >= 0
                    && role.authorizations[ftiAuthorizationIndex].write===true;

                  if (scope.currentRoleAllowEditFti){
                    // editAttributesGranted contient le nom des attributs
                    // contenus dans les autorisations du rôle de l'utilisateur
                    // les noms d'attributs sont additionnés aux éventuels autres
                    // attributs autorisés par d'autres rôles de l'utilisateur
                    const attributesAuthorizations
                      = role.authorizations[ftiAuthorizationIndex].subItems;
                    for (const attrAuthorization of attributesAuthorizations){
                      if (attrAuthorization.write){
                        scope.editAttributesGrantedSet.add(attrAuthorization.item);
                      }
                    }
                    scope.editFtiGranted = true;
                  }
                }
                scope.editAttributesGranted = Array.from(scope.editAttributesGrantedSet);
              }
            }else{
              scope.editFtiGranted = false;
            }
          }

          try {
            scope.actionsComponent = scope.currentfti.actions.map((action) => {
              if (action.currentComponent) return action;
            }).filter((x) => {
              if (x) return x;
            });
          } catch (e) {
            e.stack;
            scope.actionsComponent = [];
          }
        };



        /**
         * Zoome sur les données filtrées de la couche courante.
         * Si aucun filtre n'est défini, la fonction efface la sélection et retourne sans action.
         * Sinon, elle effectue une requête pour récupérer les données filtrées,
         * les ajoute à la sélection et ajuste la vue de la carte
         * pour afficher l'étendue des données.
         *
         * @function zoomOndata
         * @memberof layerManagerDataTable
         */
        scope.zoomOndata = () => {
          // -- Bonus - Ticket 3672:
          // --   Quand pas de filtre enlever la couche de sélection, et
          // --   ne pas tenter de filtrer.
          if (!scope.filtertemp || !scope.filtertemp.filter || scope.filtertemp.filter === '') {
            SelectManager.clear();
            return;
          }

          gaDomUtils.showGlobalLoader();
          SelectManager.clear();
          QueryFactory.data(scope.currlayer.fti.uid,
            scope.filtertemp.filter,
            scope.map.getView().getProjection().getCode(),
            '', '', ''
          ).then(
            (res) => {
              gaDomUtils.hideGlobalLoader();
              SelectManager.addFeaturesFromGeojson(res.data);
              scope.map.getView().fit(SelectManager.getExtent(), scope.map.getSize());
            },
            () => {
              gaDomUtils.hideGlobalLoader();
            }
          );
        };


        // update table height when element is resized
        scope.$on('elementResized', (event, args) => {
          if (args.element.type == 'panel' && args.element.id == 'donnees') {
            console.log(scope.layerdatatable.height);
            var currHeight =
              scope.layerdatatable.height + args.transformation.y;
            console.log(args.transformation.y);
            console.log(currHeight);
            scope.$apply((scope.layerdatatable.height = currHeight));
          }
        });


        ActionsManager.setMap(scope.map);


        initialize();
        initFromRole();

      }
    };
  };
  layerManagerDataTable.$inject = ['$rootScope', 'gclayers',
    'extendedNgDialog', 'EditFactory', '$filter',
    'SelectManager', 'gaDomUtils', 'ActionsManager',
    'QueryFactory'
  ];
  return layerManagerDataTable;
});
