'use strict';
define(function() {
  /**
   * Class : selecttoolbypointutilitiesservice
   * Factory selecttoolbypointutilities
   */
  const selecttoolbypointutilitiesservice = function(ConfigFactory, $q, $filter) {
    const selecttoolbypointutilitiesservice = {};

    /**
     * Vérifie si la parcelle est autorisée par les restrictions d'accès aux données majic de l'utilisateur.<br>
     * Renvoie true si l'utilisateur a les droits d'accès aux données majic de la parcelle:<ul><li>
     * Récupère le champ identifiant dans la configuration du widget 'majic' ('idp' par défaut)</li><li>
     * Créé le code insee de la parcelle à partir du champ identifiant</li><li>
     * Renvoie true ou false sur la présence du code insee de la parcelle dans la liste des communes de l'utilisateur</li><li>
     * Renvoie true si la liste des communes de l'utilisateur est vide</li><li></ul>
     * @param parcelle parcelle cadastrale sélectionnée par le i rouge
     * @param communes [string] liste de codes insee des communes de l'utilisateur
     */
    const userHasMajicAuthorizationOnParcelle = (parcelle, communes) => {
      let defer = $q.defer();
      ConfigFactory.get('widgets', 'majic').then(
          res => {
            getUserHasMajicAuthorizationOnParcellePromise(defer, res.data, parcelle, communes);
          },
          () => {
            getUserHasMajicAuthorizationOnParcellePromise(defer, null, parcelle,communes);
          }
      );
      return defer.promise;
    };

    /**
     * Construit la promise pour la vérification de l'autorisation d'accès aux données Majic de la parcelle
     * @param defer constructeur de la promise
     * @param config objet json du fichier de configuration
     * @param parcelle objet parcelle retourné du back
     * @param communes liste des codes insee des communes desquelles les données Majic sont accessibles à l'utilisateur
     * @return {Promise} promise contenant le boolean suivant l'appartenance du code insee de la parcelle aux communes de l'utilisateur
     */
    const getUserHasMajicAuthorizationOnParcellePromise = (defer, config, parcelle, communes) => {
      // champ identifiant de la parcelle
      const hasParcelleIdName = config && config.parcelleIdName && config.parcelleIdName.length > 0;
      if (!hasParcelleIdName) {
        require('toastr').info(
            // le champ identifiant des parcelles n'est pas configuré dans le widget 'majic'
            $filter('translate')('tools.majicedigeo.majic.noIdParcelle')
        );
      }
      const parcelleIdName = config && config.parcelleIdName && config.parcelleIdName.length > 0 ? config.parcelleIdName : 'idp';

      // vérification de l'existence du champ dans l'objet sélectionné
      const parcelleHasInsee = parcelle !== null && parcelle.hasOwnProperty('properties')
          && parcelle.properties.hasOwnProperty(parcelleIdName) && parcelle.properties[parcelleIdName].length > 6;

      // vérification de l'existence de conditions d'accès restreintes aux données majic
      const userHasMajicResctriction = communes && communes.length > 0;

      // détermine le code insee de la parcelle à partir des 6 premiers caractères du champ identifiant
      const inseeParcelle = parcelleHasInsee && parcelle.properties[parcelleIdName].substring(0, 6);

      // vérifie si l'utilisateur restreint détient les droits d'accès aux données majic de la commune de la parcelle
      const userHasCommune = parcelleHasInsee && userHasMajicResctriction && communes.includes(inseeParcelle);

      if (!userHasMajicResctriction || userHasCommune) {
        // si l'utilisateur ne possède pas de restriction d'accès aux données majic ou bien
        // si l'utilisateur a des droits d'accès aux données majic restreints et possède l'accès à la commune de la parcelle
        defer.resolve(true);
      } else {
        // clic sur la parcelle d'une commune qui ne fait pas partie des communes autorisées
        defer.resolve(false);
      }
      return defer.promise;
    };

    /**
     * Affiche un message toastr pour signaler la réussite ou l'échec de la mise à jour d'un objet
     * @param {object} backReturn objet renvoyé par le back après editFactory.update
     */
    const displayToastrAfterUpdate = (backReturn) => {
      if (backReturn.update && backReturn.update.length === 1) {
        require('toastr').success($filter('translate')('common.updated'));
      } else if (backReturn.errors && backReturn.errors.length === 1) {
        require('toastr').error($filter('translate')('elastic.search.save_nok'));
      }
    };

    return {
      selecttoolbypointutilitiesservice: selecttoolbypointutilitiesservice,
      userHasMajicAuthorizationOnParcelle: userHasMajicAuthorizationOnParcelle,
      displayToastrAfterUpdate: displayToastrAfterUpdate
    };
  };
  selecttoolbypointutilitiesservice.$inject = ['ConfigFactory','$q', '$filter'];
  return selecttoolbypointutilitiesservice;
});
