'use strict';

/**
 * Factory qui permet d'accéder à la configuration et à l'environnement de l'application Indigau
 *
 * Cette factory gère :
 * - La configuration des Feature Types (FTI) spécifiques à Indigau
 * - Les constantes et paramètres d'environnement
 * - La gestion des noms de ressources et des rapports
 *
 * @module XG/widgets/utilities/form/services/IndigauEnvFactory
 * @requires module:FeatureTypeFactory
 * @requires $rootScope
 * @requires module:ApplicationFactory
 * @requires module:ConfigFactory
 * @requires $q
 * @requires module:gaJsUtils
 */
define(function() {

  const IndigauEnvFactory = function(FeatureTypeFactory, $rootScope,
    ApplicationFactory, ConfigFactory, $q, gaJsUtils) {

    const INDIGAU_FTIS = {};
    const FTI_KEYS = [
      'ponderation',
      'indicateurs',
      'parametres',
      'critereas',
      'priority',
      'correspondance',
      'chantiers',
      'chantiers_relations'
    ];
    const resources = {};


    const setConstants = (storeName) => {
      const HPO_EXTENSION = '_' + storeName;
      resources.rapportName = 'indigau_fiche_chantier' + HPO_EXTENSION + '.docx';
    };


    /**
     * Récupère le nom du datastore à utiliser pour les features Indigau.
     * Cette fonction interroge l'application Indigau et sa configuration pour déterminer
     * le datastore approprié. Si aucun datastore n'est spécifié dans la configuration,
     * ou si une erreur se produit, le datastore par défaut 'COMMON' est utilisé.
     *
     * @returns {Promise<Object>} Une promesse qui se résout avec un objet contenant
     *                           la propriété storeName indiquant le nom du datastore à utiliser
     */
    const dataStoreName = () => {
      const defer = $q.defer();
      let mainDb = null;
      if (gaJsUtils.notNullAndDefined($rootScope.xgos, 'portal.parameters.mainDB')) {
        mainDb = $rootScope.xgos.portal.parameters.mainDB;
      }
      ApplicationFactory.get().then((res) => {
        const indigauApp = res.data.find((app) => app.type === 'Indigau');
        if (indigauApp) {
          resources.indigauAppName = indigauApp.name;
          ConfigFactory.get('main', 'appconfig', indigauApp.name).then((res) => {
            if (res && res.data && res.data.datastoreName ) {
              resources.configHpo = res.data;
              defer.resolve({storeName: res.data.datastoreName});
            }
          },
          () => {
            console.error('Impossible de récupérer la configuration '
              + 'de l\'application Indigau ' + indigauApp.name);
            if (mainDb) {
              console.error('Le datastore du portail ' + mainDb + ' est utilisé par défaut');
            }
            defer.resolve({storeName: mainDb});
          });
        } else {
          console.error('Aucune application indigau n\'a été trouvée.');
          if (mainDb) {
            console.error('Le datastore du portail ' + mainDb + ' est utilisé par défaut');
          }
          defer.resolve({storeName: mainDb});
        }
      },
      () => {
        console.error('Aucune application indigau n\'a été trouvée.');
        if (mainDb) {
          console.error('Le datastore du portail ' + mainDb + ' est utilisé par défaut');
        }
        defer.resolve({storeName: mainDb});
      });
      return defer.promise;
    };


    // -- Fabrication de la liste des FTIs des composants de travail
    // -- propres à KIS INDIGAU et récupération de la configuration
    // -- de l'application Indigau.
    dataStoreName().then((res) => {
      if (res && res.storeName) {
        const datastore = res.storeName;
        setConstants(datastore);
        for (const ftiKey of FTI_KEYS) {
          INDIGAU_FTIS[ftiKey]
              = FeatureTypeFactory.getFeatureByNameAndDatastore(datastore,
              'KIS_INDIGAU_' + ftiKey.toUpperCase() + '_' + datastore);
        }
      }
    });


    const getConfigHpo = () => {
      return resources.configHpo;
    };

    const getIndigauAppName = () => {
      return resources.indigauAppName;
    };

    const getRapportName = () => {
      return resources.rapportName;
    };


    return {
      INDIGAU_FTIS: INDIGAU_FTIS,
      FTI_KEYS: FTI_KEYS,
      indigauAppName: getIndigauAppName,
      configHpo: getConfigHpo,
      rapportName: getRapportName
    };
  };
  IndigauEnvFactory.$inject = ['FeatureTypeFactory', '$rootScope',
    'ApplicationFactory', 'ConfigFactory', '$q', 'gaJsUtils'];
  return IndigauEnvFactory;
});
