'use strict';
define(function () {
  /**
   * Class : ThemesFactory
   * Factory WebServices
   */
  var themesFactory = function (
      $http,
      $rootScope,
      UsersFactory,
      $filter,
      gaJsUtils
  ) {

    const THEME_LOCAL_VAR = 'kisusertheme'

    /**
     * Défini à null la propriété theme des users ayant un thème dans le portail
     * après que l'on ait décoché la case située dans l'onglet "Thèmes" de l'édition d'un portail
     * @param uid identifiant du portail en cours d'édition
     * @param allowUserToggleTheme est true si l'utilisateur est "root"
     * @return {boolean} true si la mise à jour a été effectuée
     */
    const removeAllUsersTheme = (uid, allowUserToggleTheme) => {
      if (allowUserToggleTheme !== undefined && allowUserToggleTheme !== null
          && allowUserToggleTheme === false) {
        let isroot = false;
        if ($rootScope.xgos && $rootScope.xgos.user && $rootScope.xgos.isroot !== undefined) {
          isroot = $rootScope.xgos.isroot;
        }
        if (localStorage.getItem(THEME_LOCAL_VAR) !== null) {
          localStorage.removeItem(THEME_LOCAL_VAR);
        }
        $http.get('/services/{portalid}/rights/user/removeAllUserThemeByPortalId?f=json' +
            '&portalid=' +
            uid +
            '&isroot=' +
            isroot
        ).then(
            res => {
              if (res.data && res.data.errors && Object.keys(
                  res.data.errors).length > 0) {
                // affiche les login des utilisateurs n'ayant pas pu être mis à jour faisant l'objet
                // d'un retour false de UserManager#updateUser
                require('toastr').error(
                    $filter('translate')('rights.users.deletethemeerror') + Object.values(
                        res.data.errors).join(', ')
                );
              }
            },
            err => {
              if (err.data && err.data.errors && Object.keys(
                  err.data.errors).length > 0) {
                require('toastr').error(
                    $filter('translate')('portals.traduction.nolanguagetocopy') + $filter(
                        'translate')('rights.users.deletethemeerror') + Object.values(
                        err.data.errors).join(', ')
                );
              }
            }
        )
      }
    }

    /**
     * Au changement du bouton toggle switch du thème,<ul><li>
     * Enregistre le nouveau thème dans l'utilisateur</li><li>
     * Met à jour l'utilisateur</li></ul>
     */
    const addDarkOrDefaultThemeToUser = () => {
      const currentTheme = $rootScope.isDarkTheme ? 'dark' : 'default';
      const applyTheme = (theme) => {
        // mise à jour du css de la page
        $rootScope.xgos.currentTheme = theme;
        // sauvegarde du thème utilisateur dans localStorage
        localStorage.setItem(THEME_LOCAL_VAR, theme);
      };

      if ($rootScope.xgos && $rootScope.xgos.user && !$rootScope.xgos.isroot) {
        // enregistrement du thème dans le user
        UsersFactory.updateUserThemeByUserId(
            $rootScope.xgos.user.uid,
            $rootScope.xgos.isroot,
            currentTheme).then(
            res => {
              if (res.data && typeof res.data === 'boolean') {
                // afectation du user global
                $rootScope.xgos.user.theme = currentTheme;
                // mise à jour du css de la page
                applyTheme(currentTheme);
              } else {
                require('toastr').error(
                    $filter('translate')('rights.users.savethemeerror')
                );
              }
            },
            err => {
              const hasDetail = err.data !== undefined && err.data.detail !== undefined;
              require('toastr').error(
                  $filter('translate')('rights.users.savethemeerror') +
                  hasDetail ? err.data.detail : ''
              );
            }
        );
      } else if ($rootScope.xgos.isroot){
        // root: mise à jour du css de la page
        applyTheme(currentTheme);
      }
    };

    /**
     * Initialise le thème de l'application puis
     * applique le thème de l'utilisateur
     * dans le cas où le portail l'autorise et dans le cas où le portail/paramètre n'existe pas.
     * Lorsque allowUserTheme = false c'est le thème du portail qui s'applique
     * @param theme theme de l'utilisateur sous forme de string
     */
    const applyUserTheme = (theme) => {
      const localTheme = localStorage.getItem(THEME_LOCAL_VAR);

      // initialise la variable locale
      if (localTheme !== null) {
        if ($rootScope.xgos.isroot) {
          // être root reprend le thème et n'efface pas la variable locale
          $rootScope.xgos.currentTheme = localTheme;
        } else {
          $rootScope.xgos.currentTheme = null;
          localStorage.removeItem(THEME_LOCAL_VAR);
        }
      }

      // vérifie la présence du portail et
      // initialise son paramètre "allowUserTheme" s'il n'existe pas
      const portalAndParamsPresent = gaJsUtils.notNullAndDefined($rootScope.xgos, ['portal', 'parameters']);
      if (portalAndParamsPresent && !$rootScope.xgos.portal.parameters.allowUserTheme) {
        $rootScope.xgos.portal.parameters.allowUserTheme = false;
      }

      // si le portail n'existe pas
      // ou s'il existe et autorise le thème utilisateur
      if (theme && (!portalAndParamsPresent || (portalAndParamsPresent
          && $rootScope.xgos.portal.parameters.allowUserTheme))) {
        // applique le thème de l'utilisateur
        $rootScope.xgos.currentTheme = theme;
        localStorage.setItem(THEME_LOCAL_VAR, theme);
      }
    };

    /**
     * Retourne le paramètre "allowUserTheme" du portail courant ou false
     * @return {boolean} true si le portail autorise l'assombrissement du thème par l'utilisateur
     */
    $rootScope.canEditTheme = () => {
      return gaJsUtils.notNullAndDefined($rootScope.xgos, ['portal', 'parameters']) ?
          $rootScope.xgos.portal.parameters.allowUserTheme : false;
    };
    $rootScope.isDarkTheme = false;

    return {
      themesFactory: themesFactory,
      removeAllUsersTheme: removeAllUsersTheme,
      addDarkOrDefaultThemeToUser: addDarkOrDefaultThemeToUser,
      applyUserTheme: applyUserTheme
    };
  };
  themesFactory.$inject = [
    '$http',
    '$rootScope',
    'UsersFactory',
    '$filter',
    'gaJsUtils'
  ];
  return themesFactory;
});