/*global define */
define(['angular', 'toastr', 'ol3js'], function(angular, toastr, ol) {
  'use strict';
  var wkt = new ol.format.WKT();

  function shareAny(a, b) {
    return a.some(function(item) {
      return b.indexOf(item) !== -1;
    });
  }

  function G2CQueryCtrl(
    $scope,
    $rootScope,
    $q,
    $timeout,
    QueryFactory,
    ParametersFactory,
    isochrone,
    gaJsUtils,
    SelectManager,
    highlightFeat,
    gcWizard
  ) {
    var query = this;
    var merlin = new gcWizard.Merlin();
    $scope.merlin = merlin;
    $scope.queries = $scope.known.queries[$scope.type];
    var listeners = {};
    var db = $scope.db;

    /*
     * Fill the scope
     */
    $scope.match = {};
    $scope.filter = {};

    angular.extend(query, {
      name: '',
      type: $scope.type,
      timeMin: $scope.known.timeMin,
      timeMax: $scope.known.timeMax,
      areaType: 'none',
      area: {},
      domains: [],
      priceMin: 0,
      priceMax: Infinity,
    });

    var errors = {
      dep: 'Veuillez sélectionner un département.',
      center: 'Veuillez sélectionner un emplacement.',
      dist: 'Veuillez sélectionner une distance valide (≤ 100 km).',
      time: 'Veuillez sélectionner un temps de trajet valide (≤ 45 min).',
      geo: 'Erreur lors de la récupération de la géométrie.',
      geofiltre: 'Vous devez choisir un filtre géographique.',
      name: 'Veuillez saisir un nom valide.',
      nameExists: 'Une requête avec le même nom existe déjà.',
      unknown: 'Erreur inconnue.',
      loadOk: 'Recherche chargée',
      saveOk: 'Enregistrement réussi.',
      deleteOk: 'Suppression réussie.',
    };

    /*
     * NAVIGATION
     */

    var steps = [
      new gcWizard.Step('home'),
      new gcWizard.Step('time'),
      new gcWizard.Step('area'),
      new gcWizard.Step('domains'),
      new gcWizard.Step('price'),
      new gcWizard.Step('save'),
    ];

    var stepList = steps.map(function(s) {
      return s.name;
    });
    function step(name) {
      return steps[stepList.indexOf(name)];
    }

    /*
     * We need to validate all of the steps before opening
     * the infobox. Step validators are expected to build the
     * results array.
     */
    $scope.openInfobox = function() {
      var defer = $q.defer();
      merlin.waiting = true;
      merlin.validateAll(function(err) {
        merlin.waiting = false;
        if (err) {
          console.error(err);
          defer.reject(err);
          return;
        }
        SelectManager.closepop();
        SelectManager.clear();
        var out = SelectManager.getfeatures();
        out.features = $scope.filter.price;
        out.totalFeatures = out.features.length;
        SelectManager.addFeaturesFromGeojson(out);
        var selectScope = $rootScope.$new(true);
        selectScope.map = $scope.map;
        selectScope.panelsManager = $scope.panelsManager;
        SelectManager.openpop(selectScope);
        defer.resolve();
      });
      return defer.promise;
    };

    $scope.loadQuery = function() {
      if (!query.newQuery) {
        return;
      }
      angular.extend(query, query.newQuery);
      delete query.newQuery; // Remove this reference
      toastr.success(errors.loadOk);
    };

    $scope.saveQuery = function() {
      var defer = $q.defer();
      function error(msg) {
        toastr.error(msg);
        defer.reject(msg);
      }
      if (!query.name) {
        error(errors.name);
        return defer.promise;
      }
      var exists = $scope.queries.some(function(qry) {
        return qry.name === query.name;
      });
      if (exists) {
        error(errors.nameExists);
        return defer.promise;
      }
      var param = $scope.param(query);
      ParametersFactory.add(query, param.type, param.name).then(
        function(res) {
          $scope.queries.push(res.data.data);
          toastr.success(errors.saveOk);
          defer.resolve();
        },
        function(res) {
          console.error(res);
          error(errors.unknown);
        }
      );
      return defer.promise;
    };

    $scope.deleteQuery = function(qry) {
      var param = $scope.param(qry);
      var promise = ParametersFactory.deletebyname(param.name);
      promise.then(
        function() {
          var index = $scope.queries.indexOf(qry);
          if (index !== -1) {
            $scope.queries.splice(index, 1);
          } else {
            console.error('Invalid index:', index);
          }
          toastr.success(errors.deleteOk);
        },
        function(res) {
          toastr.error(errors.unknown);
          console.error(res);
        }
      );
      return promise;
    };

    /*
     * STEP: TIME -----------------------------------------------
     */
    var time = step('time');

    time.prepare = function(defer) {
      query.timediff = query.timediff || 10;
      listeners.timediff = $scope.$watch('query.timediff', function() {
        query.timeMin =
          query.timediff === -1
            ? $scope.known.timeMin
            : $scope.known.year - query.timediff;
      });
      listeners.timeMin = $scope.$watch('query.timeMin', function() {
        query.timediff = $scope.known.year - query.timeMin;
      });
      defer.resolve();
    };

    time.validate = function(defer) {
      time.cql = [];
      if (query.timeMin > $scope.known.timeMin) {
        time.cql.push('(' + db.time + " >= '01-01-" + query.timeMin + "')");
      }
      if (query.timeMax < $scope.known.timeMax) {
        time.cql.push('(' + db.time + " <= '31-12-" + query.timeMax + "')");
      }
      // Unregister watches
      listeners.timediff();
      listeners.timeMin();
      defer.resolve();
    };

    /*
     * STEP: AREA -----------------------------------------------
     */
    var area = step('area');

    area.prepare = function(defer) {
      query.area.distance = query.area.distance || 50;
      query.area.time = query.area.time || 10;
      area.reqSave = area.reqSave || {};
      if (highlighted) {
        highlightFeat.remove(highlighted);
      }
      if ($scope.known.departments) {
        return defer.resolve();
      }

      // Get the list of departments
      QueryFactory.dataattribute(db.depTable, db.depName).then(
        function success(res) {
          $scope.known.departments = res.data;
          defer.resolve();
        },
        function failure(res) {
          toastr.error(errors.unknown);
          console.error('Unable to fetch department list:', res);
          defer.reject();
        }
      );
    };

    function getDepartment(name) {
      var defer = $q.defer();
      if (!name) {
        defer.reject(errors.dep);
        return defer.promise;
      }
      QueryFactory.pdata(db.depTable, db.depName + " = '" + name + "'").then(
        function success(res) {
          var geom = res.data.features[0].geometry;
          var poly = new ol.geom[geom.type](geom.coordinates, 'XY');
          defer.resolve(poly);
        },
        function failure(res) {
          defer.reject(errors.geo);
          console.error(res);
        }
      );
      return defer.promise;
    }

    function getCircle(center, dist) {
      var defer = $q.defer();
      if (!Array.isArray(center) || center.length !== 2) {
        defer.reject(errors.center);
      } else if (typeof dist !== 'number' || dist < 1 || dist > 100) {
        defer.reject(errors.dist);
      } else {
        // Create a circle (radius is in projection units, ie meters)
        var circle = new ol.geom.Circle(center, dist * 1000);
        // Turn it into a polygon to get correct coordinates
        var poly = ol.geom.Polygon.fromCircle(circle, 60);
        defer.resolve(poly);
      }
      return defer.promise;
    }

    function getIsochrone(center, time) {
      var defer = $q.defer();
      if (!Array.isArray(center) || center.length !== 2) {
        defer.reject(errors.center);
      } else if (typeof time !== 'number' || time < 1 || time > 45) {
        defer.reject(errors.time);
      } else {
        isochrone.get(center, time).then(
          function(poly) {
            defer.resolve(poly);
          },
          function() {
            defer.reject(errors.unknown);
          }
        );
      }
      return defer.promise;
    }

    /*
     * Show the desired feature on the map
     */
    var highlighted = null;
    function showFeat(feat) {
      highlightFeat.remove(highlighted);
      highlighted = highlightFeat.add(feat);

      // Animate the view *after* rendering the feature
      $timeout(function() {
        $scope.map.getView().fit(highlighted.getGeometry().getExtent(), {
          size: $scope.map.getSize(),
          padding: [0, 0, 0, 150],
          minResolution: 40,
          duration: 400,
        });
      }, 10);
    }

    area.validate = function(defer) {
      function error(msg) {
        toastr.error(msg);
        return defer.reject(msg);
      }

      function resolve() {
        if (
          query.timeMin !== area.reqSave.timeMin ||
          query.timeMax !== area.reqSave.timeMax
        ) {
          return queryAffaires(highlighted.getGeometry());
        }
        $scope.filter.area = area.resSave.features;
        defer.resolve();
      }

      function queryAffaires(poly) {
        showFeat(poly);
        var wktGeom = wkt.writeGeometry(poly);
        var cql = step('time')
          .cql.concat('INTERSECTS(geom, ' + wktGeom + ')')
          .join(' AND ');

        var crs = $scope.map
          .getView()
          .getProjection()
          .getCode();
        var promise = QueryFactory.pdata(db.table, cql, crs);

        promise.then(
          function(res) {
            area.reqSave = {
              timeMin: query.timeMin,
              timeMax: query.timeMax,
              area: angular.copy(query.area),
            };
            area.resSave = res.data;
            resolve();
          },
          function(res) {
            // Query failed
            console.log(res);
            defer.reject('Query failed');
          }
        );
        return promise;
      }

      switch (query.areaType) {
        case 'department':
          delete query.area.center;
          delete query.area.distance;
          delete query.area.time;
          if (angular.equals(query.area, area.reqSave.area)) {
            showFeat(highlighted);
            return resolve();
          }
          getDepartment(query.area.department).then(queryAffaires, error);
          break;
        case 'distance':
          delete query.area.department;
          delete query.area.time;
          if (angular.equals(query.area, area.reqSave.area)) {
            showFeat(highlighted);
            return resolve();
          }
          getCircle(query.area.center, query.area.distance).then(
            queryAffaires,
            error
          );
          break;
        case 'time':
          delete query.area.department;
          delete query.area.distance;
          if (angular.equals(query.area, area.reqSave.area)) {
            showFeat(highlighted);
            return resolve();
          }
          getIsochrone(query.area.center, query.area.time).then(
            queryAffaires,
            error
          );
          break;
        default:
          toastr.error(errors.geofiltre);
          defer.reject('Not a geographic filter: ' + query.areaType);
      }
    };

    /*
     * STEP: DOMAINS --------------------------------------------
     */
    var domains = step('domains');

    domains.prepare = function(defer) {
      var domains = [];
      $scope.filter.area.forEach(function(feat) {
        feat.properties.domains = feat.properties.departement
          .split(',')
          .map(function(dom) {
            return dom.trim();
          })
          .filter(function(dom) {
            return dom;
          });
        domains = domains.concat(feat.properties.domains);
      });
      $scope.match.domains = gaJsUtils.arrayUnique(domains).sort();
      // Only keep domains which exist in the results
      query.domains = query.domains.filter(function(dom) {
        return $scope.match.domains.indexOf(dom) !== -1;
      });
      defer.resolve();
    };

    domains.validate = function(defer) {
      if (query.domains.length > 0) {
        $scope.filter.domains = $scope.filter.area.filter(function(feat) {
          return shareAny(query.domains, feat.properties.domains);
        });
      } else {
        $scope.filter.domains = $scope.filter.area;
      }
      defer.resolve();
    };

    /*
     * STEP: PRICE ----------------------------------------------
     */
    var price = step('price');

    price.prepare = function(defer) {
      var prices = gaJsUtils
        .arrayUnique(
          $scope.filter.domains.map(function(feat) {
            return feat.properties[db.price];
          })
        )
        .sort(function(a, b) {
          return a - b;
        });

      if (prices.length > 3) {
        $scope.match.priceMin = prices.shift();
        $scope.match.priceMax = prices.pop();
        if (price.min !== undefined && query.priceMin === price.min) {
          query.priceMin = $scope.match.priceMin;
        }
        if (price.max !== undefined && query.priceMax === price.max) {
          query.priceMax = $scope.match.priceMax;
        }
        price.min = $scope.match.priceMin;
        price.max = $scope.match.priceMax;
        var steps = [1, 5, 10, 20, 50, 100, 200, 500, 1000];
        var range = $scope.match.priceMax - $scope.match.priceMin;
        $scope.match.priceStep = 5000;
        steps.some(function(step) {
          if (range / step <= 50) {
            $scope.match.priceStep = step;
            return true;
          }
        });
      } else {
        $scope.match.priceMin = 0;
        $scope.match.priceMax = 0;
        price.min = 0;
        price.max = 0;
        query.priceMin = 0;
        query.priceMax = 0;
        console.error('Not enough prices:', prices);
      }
      defer.resolve();
    };

    price.validate = function(defer) {
      if ($scope.match.priceMin === $scope.match.priceMax) {
        $scope.filter.price = $scope.filter.domains;
      } else {
        $scope.filter.price = $scope.filter.domains.filter(function(feat) {
          var montant = feat.properties[db.price];
          return montant >= query.priceMin && montant <= query.priceMax;
        });
      }
      defer.resolve();
    };

    /*
     * Add the steps to the wizard manager
     */
    steps.forEach(merlin.addStep);
    $scope.$on('$destroy', function() {
      highlightFeat.remove(highlighted);
      Object.keys(listeners).forEach(function(name) {
        listeners[name](); // unregister
      });
    });
  }

  G2CQueryCtrl.$inject = [
    '$scope',
    '$rootScope',
    '$q',
    '$timeout',
    'QueryFactory',
    'ParametersFactory',
    'isochrone',
    'gaJsUtils',
    'SelectManager',
    'highlightFeat',
    'gcWizard',
  ];

  return G2CQueryCtrl;
});
