'use strict';
define(function () {
  /**
   * Service pour l'intervention simple:<ul><li>
   * Ajuste la position de la popover du bouton de configuration</li><li>
   * Construction des labels personnalisés des pages de l'IS et édition de l'IS</li><li>
   * Construction du fil d'ariane de l'intervention simple</li><li>
   * Modification d'étapes du fil d'ariane</li><li>
   * Evaluation de la présence d'objets sélectionnés</li><li>
   * Création/suppression d'eventListeners/observers sur les champs de l'étape d'édition "Attributs"</li><li>
   * Appel api pour récupérer le nom du champ utilisé dans une association</li></ul>
   * @param $filter
   * @param gaJsUtils
   * @param $timeout
   * @param $window
   * @param $q
   * @param AssociationFactory
   * @param bizeditProvider
   * @return {{initIsBreadCrumb: ((function(Object, boolean=): (null|{currentStep: number, steps: {id: number, label: (*|string)}[]}))|*), hasRelatedOrAssocFeatures: (function(Object, Object, string)), navigationArray: {edition: number[], creation: number[]}, addIsListenersAndObservers: addIsListenersAndObservers, addAttachmentsStep: addAttachmentsStep, addSelectedObjStep: addSelectedObjStep, setDefaultBreadcrumbValues: setDefaultBreadcrumbValues, setIsDefaultLabelsAndIcons: setIsDefaultLabelsAndIcons, isForm: {isDirty: boolean}, hasSeveralStepsShown: ((function(Object[]): boolean)|*), selectedObjectsFti: {uid: string, historicColumnName: null, historicFtiUid: null, name: string, alias: string, nameAliasMapping: {Couche: string, Identifiant: string}, storeName: string, attributes: [{popup: boolean, size: number, name: string, alias: string, restrictions: *[], type: string, $$hashKey: string, isNillable: boolean, mandatory: boolean, autoCalcul: null, nillable: boolean}, {popup: boolean, size: number, name: string, alias: string, restrictions: *[], type: string, $$hashKey: string, isNillable: boolean, mandatory: boolean, autoCalcul: null, nillable: boolean}], $selected: boolean, parameters: null, srid: string}, adjustPopoverPosition: adjustPopoverPosition, isHasSelectedFeatures: (function(Object, Object)), replaceSelectedObjStep: replaceSelectedObjStep, clearFormListenersAndObservers: clearFormListenersAndObservers}}
   */
  let isUtils = function ($filter, gaJsUtils, $timeout, $window, $q, AssociationFactory, bizeditProvider) {

    // fti utile seulement pour l'affichage des colonnes "Identifiant" et "Couche" de la datatable de l'étape "Objets sélectionnés"
    let selectedObjectsFti = {
      'uid': '3a716ffb-77f4-4ffa-909e-5bae9f581f48',
      'historicFtiUid': null,
      'historicColumnName': null,
      'parameters': null,
      'name': 'selectedObjectsFti',
      'storeName': 'COMMON',
      'alias': 'selectedObjectsFti',
      'srid': 'EPSG:3857',
      'attributes': [
        {
          'name': 'Identifiant',
          'alias': 'Identifiant',
          'type': 'java.lang.Integer',
          'isNillable': false,
          'mandatory': false,
          'size': 500,
          'restrictions': [],
          'popup': false,
          'autoCalcul': null,
          'nillable': false,
          '$$hashKey': 'object:6097'
        },
        {
          'name': 'Couche',
          'alias': 'Couche',
          'type': 'java.lang.String',
          'isNillable': false,
          'mandatory': false,
          'size': 500,
          'restrictions': [],
          'popup': false,
          'autoCalcul': null,
          'nillable': false,
          '$$hashKey': 'object:6096'
        }
      ],
      'nameAliasMapping': {
        'Couche': 'Couche',
        'Identifiant': 'Identifiant'
      },
      '$selected': true,
    };

    // Ordre des id d'étapes en mode création
    // Géométrie > (Objets sélectionnés) > Attributs > (Pièces jointes)
    const defaultCreationSort = [0, 2];

    // Ordre des id d'étapes en mode édition
    // Attributs > (Pièces jointes) > Géométrie > Objets cibles
    const  defaultEditionSort = [2, 0, 1];

    const navigationArray = {
      creation: [...defaultCreationSort],
      edition: [...defaultEditionSort]
    }

    // configuration par défaut des labels des étapes d'édition (curtemplate.labels.edition.steps)
    const defaultSteps = [
        // géométrie
      {title: 'geometry', icon: 'code-fork', id: 0},
        // objets cibles
      {title: 'targetObj', icon: 'bars', id: 1},
        // attributs
      {title: 'attributes', icon: 'pencil', id: 2},
        // pièces-jointes
      {title: 'attachments', icon: 'envelope-o', id: 3},
        // objets sélectionnés
      {title: 'selectedObj', icon: 'bars', id: 4}
    ];

    // boolean témoin d'une modification de l'IS
    const isForm = {};

    // observers des champs particuliers de l'IS
    const observers = new Map();

    // associations de l'objet IS (évite de surcharger l'appel api AssociationFactory.getAFieldOrBFieldName)
    let knownAssoc = new Map();

    const ruleSelector = '.formRenderMainWrapper .its_edition_body .selection.subpart .corps .tab-content';
    const tabContainerCssClass = 'scrolledTabs';

    /**
     * Créé les valeurs par défaut de la configuration du fil d'ariane d'une IS:<ul><li>
     *   mode de création du fil d'ariane</li><li>
     *   affichage de l'onglet "Pièces-jointes</li></ul>
     * @param {object} curtemplate json du formulaire IS courant
     */
    const setDefaultBreadcrumbValues = (curtemplate) => {
      if (!curtemplate.breadcrumb) {
        curtemplate.breadcrumb = {};
        if (!curtemplate.breadcrumb.hasOwnProperty('iconMode')) {
          curtemplate.breadcrumb.iconMode = false;
        }
        if (!curtemplate.breadcrumb.hasOwnProperty('labelMode')) {
          curtemplate.breadcrumb.labelMode = true;
        }
        if (!curtemplate.breadcrumb.hasOwnProperty('hideAttachmentsStep')) {
          curtemplate.breadcrumb.hideAttachmentsStep = false;
        }
      }
    };

    /**
     * Lance la création des textes et des icones par défaut des labels personnalisables présents dans les pages de l'IS
     * @param {object} curtemplate json du formulaire IS courant
     */
    const setIsDefaultLabelsAndIcons = (curtemplate) => {
      if (!curtemplate.labels) {
        curtemplate.labels = {};
        const dfltLabelPrefix = 'tools.builder.intervention_simple.cfgPopover.defaultLabels.';

        const isPagelabels = {
          search: ['selectionRecover', 'searchFilter'],
          edition: ['objectCreation', 'objectEdition', 'targetObjList', 'selectedObjList']
        };
        // labels de l'onglet "Recherche" et autres labels de l'onglet "Edition"
        setDefaultPageLabels(curtemplate, dfltLabelPrefix, isPagelabels);

        // labels des étapes de l'onglet "Edition"
        if (!curtemplate.labels.edition.steps) {
          setDefaultEditionStepsLabels(curtemplate, dfltLabelPrefix);
        }
      }
    };

    /**
     * Créé les textes par défaut des labels personnalisables présents dans les pages de l'IS
     * @param {object} curtemplate json du formulaire IS courant
     * @param {string} dfltLabelPrefix chemin du label dans la locale (ex. 'tools.builder.intervention_simple...')
     * @param {object} isPagelabels objet contenant les noms des labels personnalisables répartis par onglet
     */
    const setDefaultPageLabels  = (curtemplate, dfltLabelPrefix, isPagelabels) => {
      for (const [pagename, inputTitles] of Object.entries(isPagelabels)) {
        if (Array.isArray(inputTitles)) {
          for (const inputTitle of inputTitles) {

            const formLabel = {
              title: inputTitle,
              label: $filter('translate')(dfltLabelPrefix + pagename + '.' + inputTitle)
            };

            if (!curtemplate.labels[pagename]) {
              curtemplate.labels[pagename] = {};
            }
            if (pagename === 'edition') {
              if (!curtemplate.labels[pagename].other) {
                curtemplate.labels[pagename].other = {};
              }
              curtemplate.labels[pagename].other[inputTitle] = formLabel;
            } else {
              if (!curtemplate.labels[pagename]) {
                curtemplate.labels[pagename] = {};
              }
              curtemplate.labels[pagename][inputTitle] = formLabel;
            }
          }
        }
      }
    };

    /**
     * Créé les icônes par défaut des labels personnalisables présents dans les pages de l'IS
     * @param {object} curtemplate json du formulaire IS courant
     * @param {string} dfltLabelPrefix chemin du label dans la locale (ex. 'tools.builder.intervention_simple...')
     */
    const setDefaultEditionStepsLabels = (curtemplate, dfltLabelPrefix) => {
      curtemplate.labels.edition.steps = [];
      for (const step of defaultSteps) {
        curtemplate.labels.edition.steps.push({
          id: step.id,
          title: step.title,
          icon: step.icon,
          label: $filter('translate')(dfltLabelPrefix + 'edition.' + step.title)
        });
      }
    };

    /**
     * Recherche une étape dans un tableau d'étapes à partir d'un entier id et renvoie le label
     * @param {object[]} steps objets contenant les labels des étapes d'édition: curtemplate.isPageLabels.edition.steps
     * @param {number} id
     * @param propertyName
     * @return {string|*}
     */
    const findIsStepConfigProperty = (steps, id, propertyName) => {
      const editStep = steps.find(step => step.id === id);
      if (editStep && editStep.hasOwnProperty(propertyName)) {
        return editStep[propertyName];
      } else {
        console.error('initBreadCrumb: aucune étape de l\'onglet Edition ne possède l\'id fourni: ',id);
        return '';
      }
    };

    /**
     * Ordonne les étapes d'édition en fonction de l'ordre des étapes dans le fil d'ariane
     * @param {object[]} steps objets contenant les labels des étapes d'édition: curtemplate.isPageLabels.edition.steps
     * @param {boolean} isNew est true quand l'objet est en cours de création, false en mode édition
     * @return {object[]} objets contenant les labels des étapes d'édition triés par ordre de présentation dans le fil d'ariane.
     * L'ordre de présentation varie entre le mode "création" et "édition"
     */
    const sortIsBreadCrumb = (steps, isNew) => {
      const sortedIdArray = isNew ? navigationArray.creation : navigationArray.edition;
      const sortedSteps = [];
      for (const id of sortedIdArray) {
        const step = steps.find(step => step.id === id);
        if (step) {
          sortedSteps.push(step);
        }
      }
      return sortedSteps;
    };

    /**
     * Initialise le fil d'ariane pour un objet en mode création ou en mode édition
     * @param {string} isMode 'render' ou 'builder'
     * @param {object} curtemplate json du formulaire IS courant
     * @param {boolean} isNew true si l'objet est en mode création (i.e current.id === undefined)
     * @param {HTMLDivElement} popupContainer conteneur principal de la popup de l'IS
     * @return {null|{currentStep: number, steps: [{id: number, label: (*|string)}, {id: number, label: (*|string)}, {id: number, label: (*|string)}, {id: number, label: (*|string)}, {id: number, label: (*|string)}]}}
     * fil d'ariane initialisé pour le cas d'un objet à créer ou pour le cas d'un objet à éditer
     */
    const initIsBreadCrumb = (isMode, curtemplate, isNew, popupContainer) => {

      setIsDefaultLabelsAndIcons(curtemplate);
      setDefaultBreadcrumbValues(curtemplate);

      const breadcrumb = curtemplate.breadcrumb;

      if (Array.isArray(curtemplate.labels.edition.steps)) {
        const labels = curtemplate.labels.edition.steps;

        if (isNew) {

          // création d'objet

          breadcrumb.steps = [
            // Géométrie
            createStep(labels, 0, true),

            // Attributs
            createStep(labels, 2, false),
          ];

          // ré-initialise l'ordre des étapes
          navigationArray.creation = [...defaultCreationSort];

          // active l'étape de géométrie
          breadcrumb.currentStep =  0;

        } else {

          // modification d'objet

          breadcrumb.steps = [
            // Attributs
            createStep(labels, 2, true),

            // Géométrie
            createStep(labels, 0, true),

            // Objets cibles
            createStep(labels, 1, true),
          ];

          // ré-initialise l'ordre des étapes
          navigationArray.edition = [...defaultEditionSort];

          if (!breadcrumb.hideAttachmentsStep) {
            // Pièces jointes
            breadcrumb.steps.splice(1, 0, createStep(labels, 3, true));
            navigationArray.edition.splice(1, 0, 3);
          }

          // active l'étape des attributs
          breadcrumb.currentStep =  2;

          // MutationObserver bug dans Chrome
          // Utiliser un timeout serait la solution
          $timeout(() => {
            addIsListenersAndObservers(popupContainer);
          }, 3000);
        }

        // adapte l'ordre des étapes (attributs d'abord)
        breadcrumb.steps = sortIsBreadCrumb(breadcrumb.steps, isNew);
        breadcrumb.goBack = false;
        breadcrumb.goNext = true;

        // KIS-3193: Lorsqu’un formulaire IS est configuré sans qu’on ait paramétré les associations patrimoines
        // alors la page “Objets cibles“ ne doit pas être affichée dans l’IS à la reprogrammation et à l'édition d’une IS existante.
        if (isMode === 'render') {
          hideTargetObjectsStepIfNoAssociation(curtemplate, breadcrumb, isNew);
        }

        return breadcrumb;
      } else {
        console.error($filter('translate')('.tools.builder.intervention_simple.cfgPopover.edition.isTitlesInitError')
            + 'curtemplate.labels = ', curtemplate.labels);
        return null;
      }
    };

    /**
     * Positionne la popover d'un bouton 'bs-popover' au-dessous de celui-ci.<br>
     * Modifie la propriété CSS 'top' dans le style inline de la popover
     */
    const adjustPopoverPosition = () => {
      $timeout(
          () => {
            const popover = document.getElementById('is-cfg-popover');
            if (popover && popover.getAttribute('show') !== 'true') {
              const bsButton = popover.previousElementSibling;
              const customCssText = [];
              const inlineStyleArray = popover.style.cssText.split(';');
              if (inlineStyleArray.length > 0 && inlineStyleArray[0].includes('top:')) {
                inlineStyleArray.shift();
              }
              const customTopValue = (bsButton.offsetTop + bsButton.clientHeight + 5) + 'px';
              customCssText.push('top: ' + customTopValue);
              const inlineCssArray = inlineStyleArray.filter(cssProp => cssProp.length > 0);
              customCssText.push(...inlineCssArray);
              popover.style.cssText = customCssText.join('; ') + ';';
              adjustPopoverLeftPosition(popover, bsButton);
              popover.setAttribute('show', 'true');
            }
          }
      );
    };

    /**
     * Aligne la popover au bord droit du bouton violet de configuration dans le cas d'un usage sur tablette<br>
     * Modifie la propriété CSS 'left' dans le style inline de la popover
     * @param {HTMLElement} popover popover en tant que élement HTML
     * @param {HTMLElement} bsButton bouton violet de configuration de l'IS
     */
    const adjustPopoverLeftPosition = (popover, bsButton) => {
      const inlineStyleArray = popover.style.cssText.split(';');
      const leftPropertyIndex = inlineStyleArray.findIndex(
          inlineCssProp => inlineCssProp.includes('left:'));
      const customCssText = [];
      for (let i = 0; i < inlineStyleArray.length; i++) {
        if (inlineStyleArray[i].length > 0) {
          if (i !== leftPropertyIndex) {
            customCssText.push(inlineStyleArray[i]);
          } else {
            const buttonBbox = bsButton.getBoundingClientRect();
            let left = buttonBbox.x;
            if ($window.innerWidth < 1200) {
              left = buttonBbox.right - popover.clientWidth;
            }
            customCssText.push('left: '+ Math.round(left) + 'px');
          }
        }
      }
      popover.style.cssText = customCssText.join('; ') + ';';
    };

    /**
     * Ajoute l'étape "Pièces-jointes" à la fin du fil d'ariane
     * @param {object[]} steps étapes du fil d'ariane de l'édition d'une IS (<code>curtemplate.breadcrumb.steps</code>)
     * @param {object[]} stepLabels labels personnalisés des étapes du fil d'ariane (<code>curtemplate.labels.edition.steps</code>)
     */
    const addAttachmentsStep = (steps, stepLabels) => {
      if (!steps.find(step => step.id === 3)) {
        steps.push(createStep(stepLabels, 3, true));
      }
    };

    /**
     * Remplace l'étape "Objets sélectionnés" par l'étape "Objets cibles"
     * @param steps étapes du fil d'ariane de l'édition d'une IS
     * @param stepLabels labels personnalisés des étapes du fil d'ariane
     */
    const replaceSelectedObjStep = (steps, stepLabels) => {
      const selectedObjIndex = steps.findIndex(step => step.id === 4);
      if (selectedObjIndex > -1) {

        // vérifie si le volet "Objets cibles" est absent
        if (!steps.find(step => step.id === 1)) {
          steps[selectedObjIndex] = createStep(stepLabels, 1, true);
        } else {
          steps.splice(selectedObjIndex, 1);
        }
        navigationArray.edition = navigationArray.edition.filter(id => id !== 4);
      }
    };

    /**
     * Vérifie la présence d'objets sélectionnés dans une variable défini dans la configuration du bouton "Enregistrer" (association ou relation)
     * @param {object} curtemplate json du formulaire IS courant
     * @param {object} res conteneur des variables du formulaire
     * @param {string} assocOrRelated nom de la propriété à vérifier dans la configuration du bouton "Enregistrer" ("association" ou "related")
     * @return {boolean} true si des objets sont stockés dans la variable fournie de l'IS courante
     */
    const hasRelatedOrAssocFeatures = (curtemplate, res, assocOrRelated) => {
      return gaJsUtils.notNullAndDefined(curtemplate.savefield, 'config.' + assocOrRelated)
          && gaJsUtils.notNullAndDefined(res[curtemplate.savefield.config[assocOrRelated]])
          && Array.isArray(res[curtemplate.savefield.config[assocOrRelated]].features)
          && res[curtemplate.savefield.config[assocOrRelated]].features.length > 0;
    };

    /**
     * Vérifie la présence d'objets sélectionnés dans l'IS courante
     * @param {object} curtemplate json du formulaire IS courant
     * @param {object} res conteneur des variables du formulaire
     * @return {boolean} true si des objets sont sélectionnés dans l'IS courante
     */
    const isHasSelectedFeatures = (curtemplate, res) => {
      return hasRelatedOrAssocFeatures(curtemplate, res, 'association') || hasRelatedOrAssocFeatures(curtemplate, res, 'related');
    };

    /**
     * Ajoute l'étape "Objets sélectionnés" au fil d'ariane d'une IS
     * @param {object} curtemplate json du formulaire IS courant
     * @param {number} insertAt index auquel insérer l'étape dans le tableau d'étapes du fil d'ariane
     * @param {boolean} show si l'étape doit être immédiatement affichée dans le fil d'ariane
     */
    const addSelectedObjStep = (curtemplate, insertAt, show) => {
      const steps = curtemplate.breadcrumb.steps;
      const labels = curtemplate.labels.edition.steps;
      if (Array.isArray(steps) && !steps.find(step => step.id === 4)) {
        const selectedObjStep = createStep(labels, 4, show);
        if (Number.isInteger(insertAt)) {
          steps.splice(insertAt, 0, selectedObjStep);
        } else {
          steps.push(selectedObjStep);
        }
      }
    };

    /**
     * Change la valeur du flag signalant une modification de l'IS (un des champs de la page "attributs")
     * {string} idDialog attribut id du conteneur principal de la popup de l'IS
     */
    const setIsDirty = (idDialog) => {
      if (!isForm[idDialog]) {
        isForm[idDialog] = {};
      }
      isForm[idDialog].isDirty = true;
      console.log('input changed');
    };

    /**
     * Ajoute un eventListener ou un observer aux champs de l'étape "Attributs"
     * contenue dans la partie "Edition" de l'IS.<br>
     * Permet la surveillance des inputs du formulaire pour détecter un changement de valeur.<br>
     * On utilise un MutationObserver pour surveiller les toggle-switch booléens, les div des associations et les inputs de calendrier.<br>
     * Les autres élements sont surveillés par un eventListener.<br>
     * @param {HTMLDivElement} popupContainer conteneur principal de la popup de l'IS
     */
    const addIsListenersAndObservers = (popupContainer) => {

      let attributeStepBody;
      let idDialog;
      if (popupContainer) {
        attributeStepBody = popupContainer.querySelector('.corps .attributs');
        idDialog = popupContainer.id;
      } else {
        attributeStepBody = document.querySelector('.corps .attributs');
        if (attributeStepBody) {
          popupContainer = attributeStepBody.closest('.popupContainer');
          if (popupContainer) {
            idDialog = popupContainer.id;
          }
        }
      }
      if (attributeStepBody && gaJsUtils.notNullAndDefined(idDialog) && idDialog.length > 0) {

        // champs avec tables de restrictions
        const allRestrictionDiv = attributeStepBody.querySelectorAll('div.ruleValue');
        const allInputs = [];
        for (const restricContainer of allRestrictionDiv) {
          const restricInputsByContainer = restricContainer.querySelectorAll('input');
          allInputs.push.apply(allInputs, restricInputsByContainer);
        }

        // Autres champs (inputs simples, inputs de calendrier, selects, associations...)
        // les champs simples reçoivent un eventListener
        // les champs particuliers sont observés (div boolean, div association, inputs de calendrier)
        const otherFormInputs = attributeStepBody.querySelectorAll('input, select, textarea, div.liste-associations, .toggle-switch-animate');

        // rassemble les champs
        allInputs.push.apply(allInputs, otherFormInputs);

        for (const inputElement of allInputs) {

          // n'ajoute pas un listener/observer à un champ déjà surveillé
          if (!inputElement.hasAttribute('eventflag')) {

            // distinction des inputs de tables de restriction et des inputs de datepicker (ces inputs ne peuvent pas être surveillés par un eventListener)
            const isRestriction = inputElement.parentNode.tagName === 'DIV' && inputElement.parentNode.classList.contains('ruleValue');
            const isCalendrier = inputElement.hasAttribute('bs-datepicker');

            // pas d'eventListener sur les champs attachment(s)
            const isAttachement = inputElement.parentNode.parentNode.classList.contains('multi-attachement')
                || inputElement.parentNode.parentNode.classList.contains('attachement');

            if (!isAttachement) {

              if (inputElement.tagName === 'DIV' || isRestriction || isCalendrier) {

                // surveille les champs particuliers par observer
                // on créé une instance par input (de peur de bugs supplémentaires sous chrome)
                const observer = new MutationObserver((mutations) => {
                  for (const mutation of mutations) {
                    if (isForm[idDialog].isInitialized) {
                      setIsDirty(idDialog);
                    }
                  }
                });

                const guid = gaJsUtils.guid();

                // ajoute un flag pour distinguer les inputs surveillés par observer
                inputElement.setAttribute('observer', guid);
                inputElement.setAttribute('eventflag', 'true');

                observer.observe(inputElement, { subtree: true, childList: true, characterData: true, attributes: true});

                // ajoute l'observer à la map de stockage pour déconnexion à la fermeture de l'IS
                observers.set(guid, observer);

              } else {
                inputElement.setAttribute('eventflag', 'true');

                // surveille les champs simples par eventListener
                inputElement.addEventListener('change', () => {
                  if (isForm[idDialog].isInitialized) {
                    setIsDirty(idDialog);
                  }
                }, false);
              }
            }
          }
        }
      }
    };

    /**
     * Arrête l'observation des champs du formulaire qui sont sous la forme de div (associations, toggle-switch).
     * Exécutée lorsque l'on quitte l'étape des attributs de l'édition d'une IS.
     * @param {string} idDialog attribut id du conteneur principal de la popup de l'IS
     */
    const clearFormListenersAndObservers = (idDialog) => {

      // récupère tous les éléments surveillés (ceux qui portent l'attribut "eventflag")
      const allInputs = document.querySelectorAll('[eventflag="true"]');

      for (const input of allInputs) {

        // supprime l'eventListener des élements qui ne sont pas surveillés par observer
        if (!input.hasAttribute('observer')) {
          input.removeEventListener('change',() => {
            setIsDirty(idDialog);
          }, false);
        }
      }

      // supprime les observers présents
      if (observers && observers.size > 0) {
        for (const key of observers.keys()) {
          if (observers.has(key)) {
            observers.get(key).disconnect();
          }
        }
      }
    };

    /**
     * Créé l'objet correspondant à une étape du fil d'ariane.
     * @param {object[]} labels configuration des labels des étapes (curtemplate.labels.edition.steps)
     * @param {number} id propriété id de l'objet étape à créer
     * @param {boolean} show propriété show de l'étape à créer
     * @return {{icon: (string|*), show: boolean, id: number, label: (string|*)}} objet prêt pour insertion dans le tableau d'étapes du fil d'ariane
     */
    const createStep = (labels, id, show) => {
      if (typeof id === 'number' && typeof show === 'boolean') {
        return {
          id: id,
          label: findIsStepConfigProperty(labels, id, 'label'),
          icon: findIsStepConfigProperty(labels, id, 'icon'),
          show: show
        }
      } else {
        console.error($filter('translate')('.tools.builder.intervention_simple.cfgPopover.edition.createStepError')
            + 'labels = ' + labels + '; id = ' + id + '; show = ' + show);
      }
    };

    /**
     * Vérifie si plusieurs étapes ont la propriété show égale à true.<br>
     * Lorsque plusieurs étapes sont visibles alors on ajoute une classe au conteneur des étapes (div de class css 'steps').<br>
     * Cette classe modifie la propriété css <code>justify-content</code> de la div (de <code>flex-start</code> vers <code>space-evenly</code>).
     * Grâce à cette classe, le fil d'ariane s'étire proprement lors du redimensionnement latéral de la popup.
     * @param {object []} steps étapes du fil d'ariane
     * @return {boolean} true si plusieurs étapes du fil d'ariane ont une propriété <code>show</code> égale à <code>true</code>
     */
    const hasSeveralStepsShown = (steps) => {
      if (Array.isArray(steps)) {
        return steps.filter(step => step.show).length > 1;
      }
    };

    /**
     * Récupère le nom du champ B de l'association reliant le composant de l'objet sélectionné au composant maître de l'IS
     * @param {string} mainDb nom de la datasource principale du portail
     * @param {string} ftiuid propriété uid du composant B
     * @param {object} selectedFti composant A
     * @param {object} selectedFeature objet sélectionné du composant A
     * @return {Promise} contient le nom du champ dont on va affficher la valeur sous la colonne "Identifiant" dans la table des objets sélectionnés
     */
    const getSelectedObjectIdentifiant = (mainDb, ftiuid, selectedFti, selectedFeature) => {
      const defer = $q.defer();

      if (knownAssoc.has(selectedFti.uid)) {

        // si le champ identifiant à afficher est déjà connu on ne fait pas l'appel API
        defer.resolve(selectedFeature.properties[knownAssoc.get(selectedFti.uid)]);

      } else if (mainDb && ftiuid) {

        AssociationFactory.getAFieldOrBFieldName(mainDb, ftiuid, selectedFti.uid).then(
            res => {
              const noResult = null;
              if (res) {

                const selectedFeatHasAssoc = typeof res.data === 'string'
                    && selectedFti.attributes.some(attr => attr.name === res.data)
                    && Object.keys(selectedFeature.properties).includes(res.data);

                const selectedFeatWithoutAssoc = res.data.hasOwnProperty('code')
                    && res.data.code === 204 && Array.isArray(res.data.details)
                    && res.data.details.length > 0;

                if (selectedFeatHasAssoc) {

                  // l'objet sélectionné fait partie d'une association patrimoine utilisable
                  knownAssoc.set(selectedFti.uid, res.data);
                  defer.resolve(selectedFeature.properties[res.data]);

                } else if (selectedFeatWithoutAssoc) {

                  // l'objet sélectionné ne fait pas partie d'une association patrimoine utilisable (ou ne fait pas partie d'une association)
                  console.info(res.data.details[0]);
                  defer.resolve(noResult);

                } else {

                  // récupère l'identifiant de l'objet même dans le cas d'un id personnalisé (ArcGIS)
                  const identifiantStr = gaJsUtils.getIdInCaseEsriId(selectedFeature, selectedFti,
                      null);
                  if (identifiantStr) {
                    // isole la partie numérique de l'identifiant (ex. "7458" pour "Troncons.7458")
                    const numAsStr = identifiantStr.split('.').slice(-1)[0];
                    defer.resolve(Number.parseInt(numAsStr));
                  } else {
                    defer.resolve(noResult);
                  }
                }
              } else {
                defer.resolve(noResult);
              }
            },
            () => {
              defer.reject();
            }
        );
      }
      return defer.promise;
    };

    /**
     * Retourne l'attribut id de l'élement form d'une intervention simple.
     * Cette recherche est effectuée pour localiser la cible d'un localLoader
     * @param childHead
     * @param dialog
     * @return {string|*}
     */
    const getLocalLoaderTarget = (childHead, dialog) => {
      if (childHead) {
        return '#is_' + childHead.$id
      } else {
        return dialog.querySelector('.renderedForm .intervention_simple');
      }
    };

    /**
     * A la reprogrammation et à l'édition d’une IS existante,
     * la page “Objets cibles“ ne doit pas être affichée dans l’IS
     * Lorsqu’un formulaire IS est configuré sans qu’on ait paramétré les associations patrimoines
     * @param curtemplate données du formulaire courant
     * @param breadcrumb fil d'ariane initialisé suivant l'état du formulaire courant (création/édition/reprog)
     * @param isNew true si le formulaire est en mode création
     * @see <a href="https://altereo-informatique.atlassian.net/browse/KIS-3193"> KIS-3193 | [DEA - IS] : cas d'une IS sans association patrimoine</a>
     */
    const hideTargetObjectsStepIfNoAssociation = (curtemplate, breadcrumb, isNew) => {
      if (!breadcrumb || !Array.isArray(breadcrumb.steps)) {
        console.error($filter('translate')('tools.builder.intervention_simple.cfgPopover.edition.breadcrumbNotInit')
            + ". breadcrumb = ", breadcrumb);
      } else {
        if (curtemplate.association && Array.isArray(curtemplate.association.associationToLoad)) {
          if (curtemplate.association.associationToLoad.length === 0) {

            // supprime l'étape "Objets cibles"
            breadcrumb.steps = breadcrumb.steps.filter(step => step.id !== 1);

            // modifie le tableau sur lequel est basé la navigation prev/forward
            if (isNew) {
              navigationArray.creation.filter(stepId => stepId !== 1);
            } else {
              navigationArray.edition.filter(stepId => stepId !== 1);
            }
          }
        } else {
          console.error($filter('translate')('tools.builder.intervention_simple.cfgPopover.edition.getAssoError')
              + ". curtemplate = ", curtemplate);
        }
      }
    };

    /**
     * Ajuste la hauteur du corps des onglets en présence de plusieurs lignes d'onglets
     * L'appel de cette méthode est effectué dans le corps de la méthode initObjetsCibles car cette dernière est censée terminer le chargement des IS
     * @param {string} iddiv
     * @param {number} triesLimit
     */
    const adjustTabContentHeight = (iddiv, triesLimit) => {
      bizeditProvider.adjustObjectFileTabContentHeight(iddiv, ruleSelector, tabContainerCssClass, triesLimit);
    };

    return {
      selectedObjectsFti: selectedObjectsFti,
      navigationArray: navigationArray,
      isForm: isForm,
      setDefaultBreadcrumbValues: setDefaultBreadcrumbValues,
      setIsDefaultLabelsAndIcons: setIsDefaultLabelsAndIcons,
      initIsBreadCrumb: initIsBreadCrumb,
      adjustPopoverPosition: adjustPopoverPosition,
      addAttachmentsStep: addAttachmentsStep,
      replaceSelectedObjStep: replaceSelectedObjStep,
      addSelectedObjStep: addSelectedObjStep,
      hasRelatedOrAssocFeatures: hasRelatedOrAssocFeatures,
      isHasSelectedFeatures: isHasSelectedFeatures,
      addIsListenersAndObservers: addIsListenersAndObservers,
      clearFormListenersAndObservers: clearFormListenersAndObservers,
      createStep: createStep,
      hasSeveralStepsShown: hasSeveralStepsShown,
      getSelectedObjectIdentifiant: getSelectedObjectIdentifiant,
      getLocalLoaderTarget: getLocalLoaderTarget,
      adjustTabContentHeight: adjustTabContentHeight
    };
  };

  isUtils.$inject = ['$filter', 'gaJsUtils', '$timeout', '$window', '$q', 'AssociationFactory', 'bizeditProvider'];

  return isUtils;
});
