'use strict';

/**
 * @ngdoc service
 * @name modules.arcgis.services:ArcGisGeoprocess
 * @description
 *   Gestion des géoprocess ArcGIS.
 *   La gestion s'occupe de récupérer la description du geotraitement, et
 *   de servir ensuite la liste des paramétres, et de récupérer le paramétre
 *   en sortie de type fichier.
 *
 */
define(function() {
  const ArcGisGeoprocess = function($filter, $q) {

    // -- Espace de travail dans lequel on stocke la description des géotraitements ArcGIS
    const workspace = {
      geoprocesses: {}
    };


    /**
     * Si la description du géotraitement est déjà stocké dans cette factory,
     * on la retourne, sinon, on l'instancie, et, l'appelant devra la remplir.
     *
     * @param {*} url  URL de base du géotraitement
     * @returns La description du géotraitement
8     */
    const getGeoprocess = (url) => {
      if (!workspace.geoprocesses[url]) {
        workspace.geoprocesses[url] = {};
      }
      return workspace.geoprocesses[url];
    };

    /**
     * Si un paramétre du géotraitement est de type paramétre en sortie et
     * que ce paramétre en sortie est un fichier, on retourne son nom.
     *
     * @param {string} url - URL de base du géotraitement.
     * @return {string} Nom du paramétre pour récupérer le fichier résultat.
     */
    const getOutParam = (url, possibleTypes, defer) => {
      // -- Créer un nouvel objet "defer" s'il n'est pas fourni
      if (!defer) {
        defer = $q.defer();
      }
      // -- Obtenir la description du service de géotraitement
      const gp = getDescription(url);
      if (gp.promise) {
        gp.promise.then(() => {
          getOutParam(url, possibleTypes, defer);
        });
        return defer.promise;
      }
      else {
        // -- Boucle sur les paramètres du service de géotraitement
        // -- et trouver celui de type "GPDataFile" (paramétre de type fichier)
        // -- avec la direction "esriGPParameterDirectionOutput"
        // -- <=> paramétre en sortie du géotraitement
        const params = Object.values(gp.description.parameters)
          .filter(param => param.direction === 'esriGPParameterDirectionOutput'
            && possibleTypes.includes(param.dataType))
          .map(param => param.name);
        defer.resolve(params.length != 0 ? params : null);
      }
      return defer.promise;
    };


    /**
     * Si un paramétre du géotraitement est de type paramétre en sortie et
     * que ce paramétre en sortie est un fichier, on retourne son nom.
     *
     * @param {string} url - URL de base du géotraitement.
     * @return {string} Nom du paramétre pour récupérer le fichier résultat.
     */
    const getOutFile = (url) => {
      return getOutParam(url, ['GPDataFile']);
    };


    const getSimpleOutParams = (url) => {
      return getOutParam(url, ['GPString','GPBoolean', 'GPDouble', 'GPlong']);
    };

    /**
     * Recupère les éléments de description du service de géotraitement ArcGIS.
     * On va récupérer la liste des paramètres.
     *
     * @param {String} url
     *    URL du géo-traitement ArcGIS
     *
     * @returns {Object}
     *    Objet avec les attributs "description" ou "promise" :
     *    - "description" si on a déjà la description du service ArcGIS
     *    - "promise" si on doit récupérer la description
     */
    const getDescription = (url) => {
      let gp;
      // -- Vérifier que l'on a bien une URL
      if (angular.isDefined(url) && url.length > 0) {
        gp = getGeoprocess(url);
      }
      // -- Si on a déjà la description, on la renvoie
      if (gp && gp.description && gp.description.parameters) {
        // -- Récupération de la description stockée quand elle a déjà été lue
        return { description: gp.description };
      }
      else {
        // -- Si on n'a pas encore la description, on la récupère
        const promise = $.get(url + '?f=json');
        promise.then((info) => {
          let parsedJsonResp;
          if(typeof info == 'string')
            parsedJsonResp = JSON.parse(info);
          else
            parsedJsonResp = info;

          if (parsedJsonResp.error) {
            gp.status = 'failed';
            require('toastr').error(
              $filter('translate')('geoprocess.errorGettingDescription') +
                    parsedJsonResp.error.message
            );
          }
          else {
            gp.description = parsedJsonResp;
          }
        });
        // -- On renvoie la promesse de récupération de la description
        return { promise: promise };
      }
    };

    return {
      getDescription, getOutFile, getSimpleOutParams
    };
  };

  ArcGisGeoprocess.$inject = ['$filter', '$q'];
  return ArcGisGeoprocess;
});
