'use strict';
define(function() {
  let MapCatalogFactory = function(
      $http,
      gaJsUtils,
      PortalsFactory
  ) {
    let MapCatalogFactory = {};

    let portalid = angular.module('gcMain').portalid;
    if (portalid === undefined || portalid === '') {
      portalid = PortalsFactory.getPortalId();
    }
    const appName = localStorage.getItem('app');
    const thumbApiUrl = '/services/' + portalid + '/mapcatalog/' + appName + '/getthumb?f=json&fileName=';
    const downloadurl = '/services/' + portalid + '/mapcatalog/' + appName + '/downloadfile?f=json&fileName=';

    /**
     * Function: getFiles
     * Méthode initiale délaissée au profit de getFilesAndConfig permettant de récupérer le titre et la catégorie
     */
    const getFiles = () => {
      return $http.get(
        '/services/{portalid}/mapcatalog/{appname}/getfiles?f=json'
      );
    };

    /**
     * Construit l'url permettant d'afficher la miniature
     * @param fileName nom du fichier avec extension (catalogItem.fileName)
     * @return {string} url d'accès au fichier dans le dossier MAPCATALOG du repo
     */
    const getExtensionLogoURL = (fileName) => {
      const fileBaseName = fileName.includes('.') ? fileName.substr(0, fileName.lastIndexOf('.')) : fileName;
      const timeNumber = Date.now();
      return (thumbApiUrl + fileBaseName + '_thb.png' + '&time=' + timeNumber);
    };

    /**
     * Au clic sur une miniature dans une dropzone, télécharge le fichier correspondant
     * @param selectedFile fichier de la miniature cliquée
     */
    const thumbnailClick = (selectedFile) => {
      const filename = selectedFile.fileName ? selectedFile.fileName : selectedFile.name;
      window.open(downloadurl + '' + filename);
    };

    /**
     * Sauvegarde les cartes dans le json du repo
     * @param mapCatalogItems tableau des objets json représentant les cartes du mapCatalog
     * @return {Promise}
     */
    const saveConfig = (mapCatalogItems) => {
      return $http.post('/services/{portalid}/mapcatalog/{appname}/saveConfig?f=json', mapCatalogItems);
    };

    /**
     * Récupère les détails des cartes du widget.<br>
     * Les propriétés filename, filesize sont déduites des images stockées dans le dossier du repo.
     * Les propriétés title et category de ces images sont stockées dans un json dans le même dossier.
     * @return {Promise} contenant un tableau d'objets [{filename,filesize,title,category},...]
     */
    const getFilesAndConfig = () => {
      return $http.get('/services/{portalid}/mapcatalog/{appname}/getFilesAndConfig?f=json');
    };

    /**
     * Supprime les cartes du repo (fichier image, miniature et objet ds json)
     * @param toDeleteMaps tableau des noms de fichiers d'image de chaque carte à supprimer
     * @return {*} tableau des cartes restantes du repo (tableau de DTO MapCatalogFile)
     */
    const deleteMaps = (toDeleteMaps) => {
      const toDelete = toDeleteMaps.join(',')
      return $http.post('/services/{portalid}/mapcatalog/{appname}/deleteMaps?f=json', toDelete);
    };

    /**
     * Supprime une carte (fichier + miniature + objet json)
     * Permet de supprimer le fichier et la miniature dans le repo depuis le catalogue
     * @param filename nom du fichier image à supprimer
     * @return {*} status 200
     */
    const removeUploaded = (filename) => {
      return $http.get('/services/{portalid}/mapcatalog/{appname}/removeUploaded?f=json&fileName=' + filename);
    };

    /**
     * Répartie les cartes par catégorie selon la propriété "category" de chaque carte
     * Si la carte n'a pas de catégorie, elle est rangée dans une catégorie nommée "défault"
     * @param catalogItems tableau de DTO de type MapCatalogFile
     * @param emptyCategoryToAdd catégorie vide à ajouter
     * @return {*[]} tableau de catégories [{name,items},...] ordonnée selon le nom asc
     */
    const buildCategories = (catalogItems, emptyCategoriesToAdd = null) => {
      const categories = [];
      if (catalogItems && catalogItems.length > 0) {

        // construit les lignes du catalogue à partir des cartes présentes dans le json
        for (const catalogItem of catalogItems) {
          const thumbnailUrl = getExtensionLogoURL(catalogItem.fileName);
          const item = {
            fileName: catalogItem.fileName,
            title: catalogItem.title && catalogItem.title !== 'null' ? catalogItem.title : catalogItem.fileName,
            thumbnail: thumbnailUrl,
            fileSize: catalogItem.fileSize,
            hasCustomTitle: catalogItem.title !== catalogItem.fileName && catalogItem.title !== 'null'
          };
          // construit les catégories à partir de la propriété 'category' de chaque carte issue du json
          // répartie les cartes dans les catégories
          if (!catalogItem.category || catalogItem.category.length === 0 || catalogItem.category === 'null') {
            catalogItem.category = 'default';
          }
          const itemCategory = categories.find(cat => cat.name === catalogItem.category);
          if (itemCategory) {
            itemCategory.items.push(item);
          } else {
            categories.push (
                {
                  name: catalogItem.category,
                  items: [item]
                }
            );
          }
        }
      }
      if (Array.isArray(emptyCategoriesToAdd) && emptyCategoriesToAdd.length > 0) {
        categories.push(...emptyCategoriesToAdd);
      }
      // tri des cartes de chaque catégorie suivant le titre asc
      for (const category of categories) {
        gaJsUtils.sortByKey(category.items, 'title');
      }
      // tri des catégorie suivant le nom asc
      if (categories.length > 0) {
        gaJsUtils.sortByKey(categories, 'name');
      }
      return categories;
    };

    /**
     * Récupère l'élément HTML de l'image miniature correspondant au nom de fichier fourni
     * @param fileName nom de fichier de la carte (fileName)
     * @return {null|Element} élément HTML portant la classe .dz-image-preview|.dz-preview
     */
    const getDzThumbByFilename = (fileName) => {
      const thumbnailFilenames = document.querySelectorAll('.dz-filename');
      for (const filenameElement of thumbnailFilenames) {
        const innerSpans = filenameElement.getElementsByTagName('span');
        if (innerSpans && innerSpans.length === 1) {
          const filename = innerSpans[0].innerText;
          if (filename === fileName) {
            let thumbnailElement = filenameElement.closest('.dz-image-preview');
            if (!thumbnailElement) {
              thumbnailElement = filenameElement.closest('.dz-preview');
            }
            if (thumbnailElement) {
              return thumbnailElement;
            }
          }
        }
      }
      return null;
    };

    /**
     * Récupère une miniature d'image en fonction de la dropzone et de son nom d'image
     * @param dropzone composant dropzone
     * @param fileName nom du fichier de la carte (fileName)
     * @return {null|Element} élément HTML portant la classe .dz-image-preview|.dz-preview
     */
    const getThumbByDropzoneAndFilename = (dropzone, fileName) => {
      const dropzoneElement = dropzone.element;
      if (dropzoneElement) {
        return getDzThumbByFilename(fileName);
      }
      return null;
    };

    const availableExtensions = [
      'aac',
      'ai',
      'aiff',
      'avi',
      'bmp',
      'c',
      'cpp',
      'css',
      'dat',
      'dbf',
      'dgn',
      'dmg',
      'doc',
      'dotx',
      'dwg',
      'dxf',
      'eps',
      'exe',
      'fix',
      'flv',
      'gif',
      'h',
      'hpp',
      'html',
      'ics',
      'iso',
      'java',
      'jpeg',
      'jpg',
      'key',
      'mid',
      'mif',
      'mp3',
      'mp4',
      'mpg',
      'odf',
      'ods',
      'odt',
      'otp',
      'ots',
      'ott',
      'pdf',
      'php',
      'png',
      'ppt',
      'prj',
      'psd',
      'py',
      'qix',
      'qt',
      'rar',
      'rb',
      'rtf',
      'shp',
      'shx',
      'sql',
      'tga',
      'tgz',
      'Thumbs',
      'tiff',
      'txt',
      'wav',
      'xls',
      'xlsx',
      'xml',
      'yml',
      'zip',
    ];

    return {
      MapCatalogFactory: MapCatalogFactory,
      getFiles: getFiles,
      getExtensionLogoURL: getExtensionLogoURL,
      thumbnailClick: thumbnailClick,
      saveConfig: saveConfig,
      getFilesAndConfig: getFilesAndConfig,
      buildCategories: buildCategories,
      deleteMaps: deleteMaps,
      removeUploaded: removeUploaded,
      getDzThumbByFilename: getDzThumbByFilename,
      getThumbByDropzoneAndFilename: getThumbByDropzoneAndFilename
    };
  };
  MapCatalogFactory.$inject = [
    '$http',
    'gaJsUtils',
    'PortalsFactory'
  ];
  return MapCatalogFactory;
});
