'use strict';
define(function () {
  /**
   * Fonction "setStaticDataToCollection" Modifie un attribut avec la valeur saisie pour l’ensemble des objets de la variable destination qui correspondent au composant sélectionné.<br>
   * <a href="https://altereo-informatique.atlassian.net/browse/KIS-3789">KIS-3789</a><ul><li>
   */
  const setStaticDataToCollectionFunction = function() {
    this.$get = function (
      $q,
      $filter,
      FeatureTypeFactory,
      QueryFactory,
      attributeRestrictionsUtils)  {
      return  {
        getParametersDesc: () => {
          return [
            [0, 'variable'],
            [1, 'componentUniqueAttributeSelector'],
            [3, 'string']
          ];
        },

      /**
       * Modifies an attribute for all objects in the destination variable that match the selected component.
       *
       * @param {object} objects - The collection of objects to modify.
       * @param {object} ftiOfCurrent - The feature type information of the current object.
       * @param {string} destination - The path to the destination attribute in the objects.
       * @param {string} composant - The component within the destination to modify.
       * @param {string} attribut - The specific attribute to set the value for.
       * @param {any} valeur - The value to set on the attribute.
       * @returns {Promise} A promise that resolves once the operation is complete.
       *
       * Checks if the destination attribute has any restrictions. 
       * If it has a table restriction, attempts to decode the value using the table.
       * If it has a domain restriction, the value is decoded via the list of values.
       * If no restriction is present, the value is directly copied.
       * Logs errors and shows notifications in case of issues.
       */

      setStaticDataToCollection: (objects, destination, composantAndAttribute, valeur) => {
        const def = $q.defer();
        let composantData = composantAndAttribute.split(';');
        let attribut = composantData.pop();
        let composant = composantData.pop();

        const features = objects.res[destination].features;
      
        if (!features || !Array.isArray(features)) {
          console.error('setStaticDataToCollection : features non trouvées ou non valides dans destination', destination);
          def.resolve();
          return def.promise;
        }
      
        const ftiOfCurrent = FeatureTypeFactory.getFeatureByUid(composant);
      
        if (!ftiOfCurrent || !Array.isArray(ftiOfCurrent.attributes)) {
          console.error('setStaticDataToCollection : ftiOfCurrent est vide ou invalide', ftiOfCurrent);
          def.resolve();
          return def.promise;
        }
      
        const destAttHasRestriction = attributeRestrictionsUtils.attributeHasRestriction(ftiOfCurrent, attribut);
      
        const setValueToAllFeatures = (valueToSet) => {
          // On filtre les features qui correspondent au composant selectionné
          const filteredFeatures = features.filter(feat => {
            const featureIdPrefix = feat.id ? feat.id.split('.')[0] : '';
            return featureIdPrefix === ftiOfCurrent.name;
          });        
          for (let feat of filteredFeatures) {
            if (!feat.properties) feat.properties = {};
            feat.properties[attribut] = valueToSet;
          }
        };
      
        if (destAttHasRestriction) {
          const hasTable = attributeRestrictionsUtils.attributeHasRestrictionType(
            ftiOfCurrent,
            attribut,
            attributeRestrictionsUtils.TABLE
          );
      
          if (hasTable) {
            // Cas table de restriction
            QueryFactory.getValueRestrictionTable(ftiOfCurrent.uid, attribut, valeur).then(
              response => {
                if (response.data && response.data.value) {
                  setValueToAllFeatures(response.data.value);
                } else {
                  require('toastr').error(
                    $filter('translate')(
                      'functions_browser.functions.setStaticDataToCollection.error_value_not_found'
                    ) + " : " + valeur
                  );
                }
                def.resolve();
              },
              err => {
                console.error('Erreur décodage via table :', err.data);
                def.resolve();
              }
            );
          } else {
            const attributDef = ftiOfCurrent.attributes.find(attr => attr.name === attribut);
            // Cas domaine de restriction (listofValues)
            if (attributDef.restrictions[0].listofValues) {
              const decoded = angular.copy(attributDef.restrictions[0].listofValues[valeur]);
              setValueToAllFeatures(decoded);
            }
            def.resolve();
          }
        } else {
          // Pas de restriction, affectation directe
          setValueToAllFeatures(valeur);
          def.resolve();
        }
      
        return def.promise;
      }
      };
    };
    this.$get.$inject = ['$q','$filter','FeatureTypeFactory','QueryFactory','attributeRestrictionsUtils'];
  };
  return setStaticDataToCollectionFunction;
});