/**
 * Gestion des fonctions liés au widget transfert de couche (TransferLayerWidget):
 *
 */
'use strict';
define(function () {
  function transferLayerFunctions () {
    this.$get = function (
      ImportTransferLayerFactory,
      $filter,
      formFunctionUtils,
      gaDomUtils
    ) {
      const LOADER_TRANSFER_LAYER = "doTransferLayerByConfigName";

      /**
       * replace token like '#$current.properties.BV' by the appropriate value contained in a form variable
       * @param {*} formVariables 
       * @param {*} dynamicFilter 
       * @returns le filtre complété sinon undefined en cas d'erreur
       */
      const fillDynamicFields = (formVariables, dynamicFilter) => {
        if (dynamicFilter.indexOf('#$') > -1) {

          // identify groups like '#$code_insee'
          const regex = /(\#\$)([^\s]+)(\s*?)/gm;

          //get an array with fields with '#$'
          const dynamicFields = dynamicFilter.match(regex);
          if (dynamicFields) {
            for (let dynamicField of dynamicFields) {
              if (typeof dynamicFilter === 'string') {
                let value = formFunctionUtils.getDataValueByKey(formVariables, dynamicField.replace('#$', ''));
                if (value != undefined) {
                  dynamicFilter = dynamicFilter.replace(dynamicField, value);
                } else {
                  dynamicFilter = undefined;
                  require('toastr').error(
                    $filter('translate')('functions_browser.functions.copyDataFromWebService.field_is_missing')
                    + '(' + dynamicField + ')');
                }
              }
            }
          }
        }
        return dynamicFilter;
      }

      /**
       * trigger the backend service
       * @param {*} resolve promise resolve function
       * @param {*} reject promise reject function
       * @param {*} configName name of the config file
       * @param {*} newFilter filtre that will eventually replace the one in the config
       */
      const startTransferLayer = (resolve, reject, appName, configName, newFilter) => {
        gaDomUtils.showGlobalLoader(LOADER_TRANSFER_LAYER);
        ImportTransferLayerFactory.doTransferLayerByConfigName(appName, configName, newFilter
          ).then(res => {
            require('toastr').success(Object.keys(res.data));
            resolve();
          }, error => {
            if (typeof error.data === 'string') {
              require('toastr').error(error.data);
            } else {
              require('toastr').error($filter('translate')('functions_browser.functions.copyDataFromWebService.error_copy'));
            }
            reject();
          }).finally(() => {
            gaDomUtils.hideGlobalLoader(LOADER_TRANSFER_LAYER);
          });
      };

      /**
       * Retourne les points d'entrée de ce provider:
       */
      return {
        //parametres de la formFonction
         getParametersDesc: () => {
          return [
            [0, 'transferLayerConfigSelector'],
            [1, 'string']
          ];
        },
        /**
         * copyDataFromWebService:
         * Appel la fonction tranfert de couche si l'utilisateur confirme
         * @param {*} formVariables variables of the form
         * @param {*} appNameAndConfigName name of the app and name of the config file separeted by |
         * @param {*} newFilter filtre that will eventually replace the one in 
         * @returns 
         */
         confirmTransferLayer: (formVariables, appNameAndConfigName, newFilter) => {
          const promise = new Promise((resolve, reject) => {
            const appName = appNameAndConfigName.includes('|') ? appNameAndConfigName.split('|')[0] : undefined;
            const configName = appNameAndConfigName.includes('|') ? appNameAndConfigName.split('|')[1] : undefined;
            if (appName && configName && newFilter != undefined) {
              ImportTransferLayerFactory.getComponentsAliasesFromConfigName(appName, configName).then(
                res => {
                  const alertText = $filter('translate')('functions_browser.functions.copyDataFromWebService.confirm_transfer')
                    + (res.data && res.data.sourceAlias ? res.data.sourceAlias : 'source')
                    + $filter('translate')('functions_browser.functions.copyDataFromWebService.to')
                    + (res.data && res.data.destinationAlias ? res.data.destinationAlias : 'destination')
                    + ' ?';

                  swal({
                    title: alertText,
                    type: 'info',
                    showCancelButton: true,
                    confirmButtonColor: '#5cb85c',
                    confirmButtonText: $filter('translate')('common.yes'),
                    cancelButtonText: $filter('translate')('common.no'),
                    closeOnConfirm: true,
                  },
                  isConfirm => {
                    if (isConfirm) {
                      newFilter = fillDynamicFields(formVariables, newFilter);
                      //Si le filtre n'a pas pu être complété, on ne lance pas le tranfert de couche.
                      if (typeof newFilter === 'string') {
                        startTransferLayer(resolve, reject, appName, configName, newFilter);
                      }
                    } else {
                      reject();
                    }
                  });
                });
              }
          });
          return promise;
        }
      };
    };
    this.$get.$inject = [
      'ImportTransferLayerFactory',
      '$filter',
      'formFunctionUtils',
      'gaDomUtils'
    ];
  };
  return transferLayerFunctions;
});
