/** A control with scroll-driven navigation to create narrative maps
 *
 * @constructor
 * @extends {ol.control.Control}
 * @fires 
 * @param {Object=} options Control options.
 *	@param {String} options.className class of the control
 *	@param {Element | string | undefined} options.html The storymap content
 *	@param {Element | string | undefined} options.target The target element to place the story. If no html is provided the content of the target will be used.
 */
ol.control.Storymap = function(options) {
  // Remove or get target content 
  if (options.target) {
    if (!options.html) {
      options.html = options.target.innerHTML;
    } else if (options.html instanceof Element) {
      options.html = options.html.innerHTML;
    }
    options.target.innerHTML = '';
  }
  var element = ol.ext.element.create('DIV', {
    className: (options.className || '') + ' ol-storymap'
      + (options.target ? '': ' ol-unselectable ol-control')
      + (ol.has.TOUCH ? ' ol-touch' : ''),
    html: options.html
  });
  // Initialize
  ol.control.Control.call(this, {
    element: element,
    target: options.target
  });
  // Make a scroll div
  ol.ext.element.scrollDiv (this.element, {
    vertical: true,
    mousewheel: true
  });
  // Scroll to the next chapter
  var sc = this.element.querySelectorAll('.ol-scroll-next');
  sc.forEach(function(s) {
    s.addEventListener('click', function(){ 
      var chapter = this.element.querySelectorAll('.chapter');
      var scrollto = s.offsetTop;
      for (var i=0, c; c=chapter[i]; i++) {
        if (c.offsetTop > scrollto) {
          scrollto = c.offsetTop;
          break;
        }
      }
      this.element.scrollTop = scrollto;
    }.bind(this));
  }.bind(this));
  // Scroll top 
  sc = this.element.querySelectorAll('.ol-scroll-top');
  sc.forEach(function(i) {
    i.addEventListener('click', function(){ 
      this.element.scrollTop = 0;
    }.bind(this));
  }.bind(this));
  // Handle scrolling
  var currentDiv = this.element.querySelectorAll('.chapter')[0];
  setTimeout (function (){
    this.dispatchEvent({ type: 'scrollto', start: true, element: currentDiv, name: currentDiv.getAttribute('name') });
  }.bind(this));
  // Trigger change event on scroll
  this.element.addEventListener("scroll", function() {
    var current, chapter = this.element.querySelectorAll('.chapter');
    var height = ol.ext.element.getStyle(this.element, 'height');
    if (!this.element.scrollTop) {
      current = chapter[0];
    } else {
      for (var i=0, s; s=chapter[i]; i++) {
        var p = s.offsetTop - this.element.scrollTop;
        if (p > height/3) break;
        current = s;
      }
    }
    if (current && current!==currentDiv) {
      currentDiv = current;
      this.dispatchEvent({ type: 'scrollto', element: currentDiv, name: currentDiv.getAttribute('name') });
    }
  }.bind(this));
};
ol.inherits(ol.control.Storymap, ol.control.Control);
/** Scroll to a chapter
 * @param {string} name Name of the chapter to scroll to
 */
ol.control.Storymap.prototype.setChapter = function (name) {
  var chapter = this.element.querySelectorAll('.chapter');
  for (var i=0, s; s=chapter[i]; i++) {
    if (s.getAttribute('name')===name) {
      this.element.scrollTop = s.offsetTop;
    }
  }
};
