'use strict';
define(function () {
  const itv = function ($filter, itvSoumFactory, DataStoreFactory, NetworkFactory,
    FeatureTypeFactory, gaDomUtils, $timeout, SridFactory, ngDialog, $rootScope) {
    return {
      templateUrl: 'js/XG/widgets/mapapp/LITV/views/itvConfigDir.html',
      restrict: 'AE',

      link: function (scope) {
        // Get configs
        if (!scope.ConfigName && scope.$parent.ConfigName) {
          scope.ConfigName = scope.$parent.ConfigName;
        }
        else if (scope.$parent.apptype == 'indigau') {
          scope.ConfigName = 'indigauItvCfg';
        }
        // -- Liste des FTI définis comme branchements dans la config réseau
        scope.connectionPipes = [];

        scope.itvTabs = {
          tabs: [
            {
              title: $filter('translate')('itv.soumission.conf.general'),
            },
            {
              title: $filter('translate')('itv.soumission.conf.params'),
            },
            {
              title: $filter('translate')('itv.soumission.conf.indigau'),
            },
            {
              title: $filter('translate')('itv.soumission.conf.advanced'),
            }
          ],
          activeTab: 0
        };
        /**
         *
         * Vérification initial du contenu des noms de champ
         * pour le nombre de défaut continu et pour le nombre de branchements
         * afin d'afficher le message pertinent en dessous de la zone de saisie.
         */
        const fieldExistenceWhenInitializing = () => {
          if (!scope.ftis || !getNetWorkEdgeFtis()) {
            $timeout(fieldExistenceWhenInitializing);
          }
          else {
            scope.fieldExists('atLeastOne', 'connectionCountField');
            scope.fieldExists('all', 'defaultCountField');
          }
        };


        /**
         * La copie de la configuration depuis le résultat de "getconf" a pour but
         * la prise en compte en direct de ce qui vint d'âtre sauvegardé quand on ferme
         * puis que l'on ouve lewidget de configuration. Sans cela, c'est à dire,
         * avec une affectation sur scope.conf direct, les scope enfanbt ne voit pas
         * les nouvelles valeurs, puisque scope.conf référencé est un autre scope.conf.
         * -- Note: scope.conf est hérité du scope parent (widget ITV) et a déjà été initialisé,
         * -- -- -- il n'est donc pas nécessaire de traiter le cas où scope.conf serait undefined.
         * @param {*} conf Configurration récupérée à mettre en place
         */
        const copyConf = (conf) => {
          if (!scope.conf) {
            scope.conf = {};
          }
          for (const prop in conf) {
            scope.conf[prop] = angular.copy(conf[prop]);
          }
        };

        const getConf = () => {
          itvSoumFactory.getConf(scope.ConfigName).then((conf) => {
            copyConf(conf);
            scope.isArcGISDataStore();
            if (scope.$parent.apptype == 'indigau') {
              // -- Le nom du réseau est à forcer quand l'application
              // -- est une application KIS INDIGAU.
              scope.conf.config.network = 'xx_itv_altereo_ass_network_xx';
            }
            fieldExistenceWhenInitializing();
            // -- Configuration du widget configuration en fonction
            // -- du contenu de la configuration
            checkConnectionPipe();
            initManholeIdRules();
            scope.conf4mod = angular.copy(scope.conf);
          });
        };

        // Datastores
        DataStoreFactory.get().then(function () {
          scope.datastores = DataStoreFactory.resources.datastores;
          // Networks
          if (NetworkFactory.resources.networks.length > 0) {
            scope.networks = NetworkFactory.resources.networks;
            getConf();
          }
          else {
            NetworkFactory.get().then(function () {
              scope.networks = NetworkFactory.resources.networks;
              getConf();
            });
          }
        });

        /**
         * Génère les ftis pour l'autocomplétion des champs components
         */
        const getFtis = () => {
          if (!scope.ftis) {
            // à la 1ère exécution après l'ouverture de la config
            FeatureTypeFactory.get().then(
              res => {
                // construit le tableau de fti exploitables par l'autocomplete
                scope.ftis = res.map(fti => {
                  return {
                    name: fti.name,
                    alias: fti.alias,
                    fti
                  };
                });
                // ouverture de la popup
              },
              (err) => {
                console.log(err);
                // ouverture de la popup dans un état non opérationnel
              }
            );
          }
        };

        getFtis();

        /**
         * Check if the selected datastore is arcgis for components display
         */
        scope.isArcGISDataStore = () => {
          if (scope.conf && scope.conf.config) {
            const selectedDatastore
              = scope.datastores.find(datastore => datastore.name === scope.conf.config.datastore);
            scope.isArcGISDatastore = selectedDatastore && selectedDatastore.type === 'ArcGis';
          }
        };

        /**
         * Au clic sur le bouton "Enregistrer" de la popup de configuration:
         * Vérifie la présence des composants
         * Propose le choix du srid pour les composants manuants
         * Lance la procédure de sauvegarde de la configuration après sélection du srid
        */
        scope.initItvConfigSave = function () {

          gaDomUtils.showGlobalLoader();
          // -- On met en place les nouvelles valeurs
          if (!scope.conf) {
            scope.conf = {};
          }
          for (const prop in scope.conf4mod) {
            scope.conf[prop] = scope.conf4mod[prop];
          }

          itvSoumFactory.checkItvConf(scope.conf, null).then(
            res => {
              if (hasMissingComponent(res.data)) {
                gaDomUtils.hideGlobalLoader();
                const newScope = $rootScope.$new();
                newScope.srids = SridFactory.sridsList;
                newScope.selectedsrid = {
                  name: scope.map.getView().getProjection().getCode()
                };
                newScope.submitSrid = () => {
                  if (newScope.dialog) {
                    newScope.dialog.close();
                  }
                  initialiseDbAndSaveConfig(newScope.selectedsrid.name);
                  newScope.$destroy();
                };
                newScope.dialog = ngDialog.open({
                  template: 'js/XG/widgets/mapapp/LITV/views/modals/itvConfig.srid.html',
                  className: 'ngdialog-theme-plain nopadding noclose itv-config-srid',
                  closeByEscape: false,
                  closeByDocument: false,
                  scope: newScope,
                });
              } else {
                // aucun composant manquant: sauvegarde directe de la configuration
                initialiseDbAndSaveConfig();
              }
            }
          );
        };

        /**
         * Effectue la création des composants ITV requis s'il n'existent pas
         * puis sauvegarde la configuration
         * @param {string|null} srid code de la projection dans laquelle définir les composants
         */
        const initialiseDbAndSaveConfig = (srid = null) => {
          gaDomUtils.showGlobalLoader();
          const datastore = scope.conf.config.datastore;
          itvSoumFactory.initDb(datastore, scope.conf, srid).then(
            (res) => {
              if (res.data.etat === 'fini') {
                require('toastr').success(
                  $filter('translate')('itv.soumission.dbsucces')
                );
              }
              else if (res.data.etat === 'erreur') {
                require('toastr').error(
                  $filter('translate')('itv.soumission.dberror')
                );
              }
              else if (res.data.etat === 'alerte') {
                let msg = '';
                const msgs = res.data.warningsAsString.split('\n');
                for (let ii = 0; ii < msgs.length; ii++) {
                  if (msg != '') msg += '\n';
                  msg += $filter('translate')(msgs[ii]);
                }
                require('toastr').warning(msg);
              }
              gaDomUtils.hideGlobalLoader();
            });

          itvSoumFactory.saveconf(scope.conf, scope.ConfigName);
        };

        /**
         * Récupére la liste des FTIs des liénaires du réseau choisi
         * pour les ITVs.
         *
         * @returns Tableau des FTIs des linéaires du réseau d'assainissement
         */
        const getNetWorkEdgeFtis = () => {
          const edgeFtis = [];
          // -- Récupération de la description du réseau d'assainissement
          const theNetwork = scope.networks.find(
            network => network.name === scope.conf.config.network
          );
          if (theNetwork !== undefined) {
            // -- Recherche des FTIs complet des liéneaires du réseau
            for (const edgeType of theNetwork.edgesTypes) {
              const fti = scope.ftis.find(fti => fti.fti.uid === edgeType.ftiUID);
              if (fti !== undefined) {
                edgeFtis.push(fti);
              }
            }
          }
          return edgeFtis;
        };

        /**
         * Vérifier que le champ saisi dans "scope.conf.extraconfig[property]
         * rempli les conditions d'existence et de type.
         *
         * Les champs pointés par les propriétés "connectionCountField" et
         * "defaultCountField" doivent être numérique.
         *
         * Quand la propriété est "connectionCountField" le nom du champ
         * que cette propiété contient doit apparaître dans au moins
         * un linéaire du réseau. On ne va pas compter les branchements
         * connectés aux branchements car ça n'existe pas.
         *
         * Quand la propriété est "defaultCountField" le nom du champ
         * que cette propiété contient doit apparaître dans dans tous les
         * linéaires du réseau. Tout linéaire inspecté est susceptible
         * d'avoir des défauts, sinon on ne les inspecterait peut-être pas..
         *
         * @param {*} where Indique si tous les linéaire doivent être porteur
         *                  du champ, ou si quand au moins un
         *                  en est porteur cela est suffisant.
         * @param {*} property Propriété de "scope.conf.extraconfig"
         *                     qui contient le nom du champ à vérifier.
         */
        scope.fieldExists = (where, property) => {
          const ftis = getNetWorkEdgeFtis();
          let attCount = 0;
          scope[property + 'Numeric'] = true;
          if (scope.conf.extraconfig) {
            for (const fti of ftis) {
              const att = fti.fti.attributes.find(
                att => att.name === scope.conf.extraconfig[property]);
              if (att) {
                if (att.type !== 'java.lang.Integer'
                    && att.type !== 'java.lang.Double') {
                  scope[property + 'Numeric'] = false;
                }
                ++attCount;
              }
            }
          }
          if (attCount === 0) {
            // -- Champs inexistant
            scope[property + 'Numeric'] = undefined;
            scope[property + 'OK'] = 'inexistant';
          }
          else if (attCount === ftis.length) {
            scope[property + 'OK'] = 'existePartout';
          }
          else {
            scope[property + 'OK'] = 'existeSurCertains';
          }
        };

        /**
         * Vérification quant au fait qu'un nom de champ soit donné pour
         * le nombre de défauts ou le nombre de branchements
         * (selon ce qui est désigné par le paramétre "property").
         *
         * @param {*} property "connectionCountField" ou "defaultCountField"
         * @returns VRAI si une valeur est définie, FAUX sinon
         */
        scope.valueIsSet = (property) => {
          return scope.conf && scope.conf.extraconfig
            && scope.conf.extraconfig[property] !== undefined
            && scope.conf.extraconfig[property].length !== 0;
        };

        /**
         * Vérifie que le réseau de cette configuration d'ITV contient
         * un ou des Feature Type Info configurés comme étant des branchements.
         * Si tel est le cas fabrication de la liste des composants branchements
         * pour la sélection par l'opérateur du composant destination
         * de la configuration de l' "extraconfig" qui permet d'utiliser
         * le code AAT comme un identifiant de branchement.
         */
        const checkConnectionPipe = () => {
          // -- Y a-t-il des composants branchement dans la configuratio ITV ?
          let currentNetwork;
          if (scope.conf) {
            currentNetwork = scope.networks.find(
              network => network.name === scope.conf.config.network);
          }
          if (!currentNetwork) {
            return;
          }
          const connectionPipeTypes = currentNetwork.connectionLinearTypes;
          scope.hasConnectionPipe = connectionPipeTypes
            && connectionPipeTypes.length !== 0;
          if (scope.hasConnectionPipe) {
            // -- Il y a des composants branchements, on fabrique la liste
            // -- pour le choix dans l'IHM.
            scope.connectionPipes.splice(0);
            for (const connectionPipe of connectionPipeTypes) {
              const fti = FeatureTypeFactory.getFeatureByUid(
                connectionPipe.ftiUID);
              scope.connectionPipes.push({ name: fti.name, fiUID: fti.uid });
            }
            // -- Composant par défaut : le premier de la liste
            scope.selectedConnectionPipe = scope.connectionPipes[0];
          }
          else {
            // -- Pas de linéaire de branchement configuré, donc
            // -- le changemeent de codification est impossible.
            scope.aatConfiguration = false;
          }
          if (scope.conf.extraconfig) {
            scope.aatConfiguration = scope.conf.extraconfig.specificEquipementCode != undefined;
          }
          else {
            scope.aatConfiguration = false;
          }
        };

        /**
         * L'utilisateur a choisi un autre réseau, on actualise les éléments
         * pour le paramétrage de l'extraconfig.
         */
        scope.networkSelected = () => {
          checkConnectionPipe();
        };

        /**
         * Mettre ou enlever la configuration qui indique que le code AAT
         * contient les identifiants des branchements au lieu de celui
         * de la boîte de raccordment.
         */
        scope.setUnsetAatConfig = () => {
          if (scope.aatConfiguration) {
            // -- Mettre la configuration spécifique
            if (!scope.conf.extraconfig) {
              scope.conf.extraconfig = {};
            }
            scope.conf4mod.extraconfig.specificEquipementCode = {
              reversenodes: true,
              component: scope.selectedConnectionPipe.name, // 'ASS_BRCHT',
              code_device_dest: '@01',
              code_device: 'AAT'
            };
          }
          else {
            // -- Enlever la configuration spécifique
            if (scope.conf.extraconfig) {
              delete scope.conf4mod.extraconfig.specificEquipementCode;
            }
          }
        };

        /**
         * Initialisation des propriétés du scope nécessaires
         * à la configuration des règles de suppression de préfiwe ou
         * de morceau de chaîne de caractères (regex) de l'identifiant
         * lue dans le fichiet ITV.
         */
        const initManholeIdRules = () => {
          // -- S'assurer de la présence de l'extraconfig et de la table rm_regex
          if (!scope.conf4mod.extraconfig) {
            scope.conf4mod.extraconfig = {};
          }
          if (!scope.conf4mod.extraconfig.manholeIdRules) {
            scope.conf4mod.extraconfig.manholeIdRules = [];
          }
          if (!(scope.conf4mod.extraconfig.manholeIdRules instanceof Array)) {
            scope.conf4mod.extraconfig.manholeIdRules = [];
          }
          scope.manholeIdRulesIhm = [];
          for (const rule of scope.conf4mod.extraconfig.manholeIdRules) {
            if (rule.rm_regex) {
              scope.manholeIdRulesIhm.push(
                {
                  type: 'expression',
                  value: rule.rm_regex
                }
              );
            }
            else if (rule.rm_prefix) {
              scope.manholeIdRulesIhm.push(
                {
                  type: 'prefix',
                  value: rule.rm_prefix
                }
              );
            }
          }
        };

        /**
         * Ajouter l'expression régulière saisie dans la zone de texte.
         */
        scope.addRegularExpression = () => {
          // -- Ajouter la regex saisie dans la liste des regex correspndants
          // -- à du contenu à enlever de l'identifiant des rgerads
          scope.conf4mod.extraconfig.manholeIdRules.push({ rm_regex: scope.expressionToAdd });
          scope.manholeIdRulesIhm.push({
            type: 'expression',
            value: scope.expressionToAdd
          });
        };

        /**
         * L'utilisateur a choisi le type de l'expression d'indice "index".
         * On prend en compte ce changement.
         * Le type de l'expression est "rm_regex" ou "rm_prefix" pour enlever
         * de l'identifiant un préfixe ou ce qui correspond à
         * une expression régulière.
         * @param {*} index Index de l'expression à traiter
         */
        scope.setExpressionType = (index) => {
          let srcType, destType;
          if (scope.manholeIdRulesIhm[index].type === 'prefix') {
            srcType = 'rm_regex';
            destType = 'rm_prefix';
          }
          else {
            srcType = 'rm_prefix';
            destType = 'rm_regex';
          }
          scope.conf4mod.extraconfig.manholeIdRules[index][destType]
            = scope.conf4mod.extraconfig.manholeIdRules[index][srcType];
          delete (scope.conf4mod.extraconfig.manholeIdRules[index][srcType]);
        };

        /**
         * Enelver une regex de la liste des rm_regex de l'extraconfig.
         *
         * @param {*} index Indice de la regex à enlever
         */
        scope.delRegularExpression = (index) => {
          scope.conf4mod.extraconfig.manholeIdRules.splice(index, 1);
          scope.manholeIdRulesIhm.splice(index, 1);
        };

        const hasMissingComponent = (data) => {
          const ERR_MSG = 'itv.arcgis_missing_components';
          return (Object.prototype.hasOwnProperty.call(data, 'warningsAsString')
              && data.warningsAsString.length > 0
              && data.warningsAsString.includes(ERR_MSG)) ||
              (Object.prototype.hasOwnProperty.call(data, 'errorList') && data.errorList.length > 0
                && data.errorList.some(err =>
                  Object.prototype.hasOwnProperty.call(err, 'message_kis')
                && err.message_kis === ERR_MSG));
        };
      }
    };
  };

  itv.$inject = ['$filter', 'itvSoumFactory', 'DataStoreFactory',
    'NetworkFactory', 'FeatureTypeFactory', 'gaDomUtils', '$timeout',
    'SridFactory', 'ngDialog', '$rootScope'
  ];
  return itv;
});
