'use strict';
/**
 *  2 parties:
 * - Configuration
 * - Utilisation
 *
 * - Administration
 *   Les fichiers du modèles sont entreposées dans repo/FILES.
 *   Le descriptif des modèles est entreposé dans repo/PRINT/ATLAS/atlas.conf
 */
define(function() {
  const atlas = function (
    AtlasFactory,
    $filter,
    gclayers,
    gaDomUtils,
    $interval,
    $q,
    $rootScope,
    BaseMapFactory,
    $location,
    $translate,
    AtlasService,
    ngDialog,
    ConfigFactory,
    gcStyleFactory,
    $timeout,
    DataStoreFactory,
    pv2LayerServices,
    gaJsUtils,
    PrintLegendFactory,
    extendedNgDialog, AtlasArcGIS108Service,
    PortalsFactory
  ) {
    return {
      templateUrl: 'js/XG/widgets/mapapp/atlas/views/atlas.html',
      restrict: 'A',
      // scope: {
      //   printAtlasCall: '=?',
      //   atlasEsriConfig: '=atlasEsri?'
      // },
      link: function(scope, elt, attrs) {
        let parsedJsonAtlasResp;

        scope.atlas = {
          legend: true,
          visibleonscreen: 'asVisibleOnMap',
          mode: 'normal',
          print: {
            rotation: -((scope.map.getView().getRotation() * 180.0) / Math.PI),
            units:
                scope.map
                  .getView()
                  .getProjection()
                  .getUnits() || 'm',
            srs: scope.map
              .getView()
              .getProjection()
              .getCode(),
            lang: $rootScope.xgos.portal.lang,
            app: $rootScope.xgos.portal.uid + '.sommaire.json',
            legend: true,
          }
        };



        scope.jobInfoStatus = '';
        scope.executingJob = false;
        scope.jobDetailMessages = [];
        scope.arcgisJobName = 'Atlas';

        scope.esriConfig = {};

        if (!scope.atlasEsri) {
          scope.atlasEsri = {
            type: 'area',
            line_models: [],
            area_models: [],
            line_formats: [],
            area_formats: [],
          };
        }

        scope.atlasTitleHasBeenProvided = scope.atlasDescHasBeenProvided = false;
        if (attrs.dicttitle) {
          scope.atlasEsri.title = attrs.dicttitle;
          scope.atlasTitleHasBeenProvided = true;
        }
        if (attrs.dictdescription) {
          scope.atlasEsri.description = attrs.dictdescription;
          scope.atlasDescHasBeenProvided = true;
        }

        var sourceFeature = new ol.source.Vector({
          //create empty vector
        });

        scope.SelectionLayer = new ol.layer.Vector({
          source: sourceFeature,
          style: gcStyleFactory.getStyle('select'),
        });
        scope.map.getLayers().insertAt(15, scope.SelectionLayer);

        scope.extentnottotakeintoaccount = [];

        var firstScale = true;
        scope.selection = {};
        var pdfLegendString = '_big.pdf';
        var printRectangle;
        var deregister;
        var DEVICE_PIXEL_RATIO = 1;
        var POINTS_PER_INCH = 72; //PostScript points 1/72"
        var MM_PER_INCHES = 25.4;

        AtlasService.setMap(scope.map);


        AtlasFactory.getPrintInfo().then(
          function(res) {
            scope.capabilities = res.data;
            scope.atlas.print.dpi = Number(scope.capabilities.dpis[0].value);
            scope.scales = res.data.scales;
            console.info('print config uploaded');
          },
          function() {
            console.error('no print info');
          }
        );

        scope.atlasUrlInput = {};


        let getModelsFromConfig = (data) => {
          if (scope.atlas_models.length > 0) {
            scope.atlas_models_configured = data.atlas_models_configured;

            if (data.atlas_models) {
              for (let i = 0; i < scope.atlas_models.length; i++) {
                for (let j = 0; j < data.atlas_models.length; j++) {
                  if (scope.atlas_models[i].id === data.atlas_models[j].id) {
                    scope.atlas_models[i].added = data.atlas_models[j].added;
                    if (data.atlas_models[j].added)
                      scope.atlas_models_validated.push(scope.atlas_models[i]);
                    break;
                  }
                }
              }
            }
          }
        };


        AtlasFactory.getconf().then(
          (resAtlasConf) => {
            ConfigFactory.get('widgets', scope.ConfigName).then(function(res) {
              if (resAtlasConf.data.etat === 'fini') {
                scope.all_models = resAtlasConf.data.data || [];
                scope.atlas_models = scope.all_models;
                /*scope.atlas_models = scope.all_models.filter(model => model.type !== 'arcgis');
                  */
                if (scope.atlas_models.length > 0) {
                  scope.atlas_models_validated = [];
                }
              }
              if (res.data !== '') {
                let data = res.data;

                if (angular.isDefined(data['atlas_ds'])) {
                  scope.esriConfig.datasource = data['atlas_ds'];
                }
                loadDataStores();

                if (angular.isDefined(data['atlas_mode'])) {
                  scope.atlas.mode = data['atlas_mode'];
                  scope.atlasMode = scope.atlas.mode;
                }

                if (resAtlasConf.data.etat === 'fini') {
                  getModelsFromConfig(data);
                } else {
                  scope.atlas_models = [];
                  console.error('no atlas config');
                }

                if (data.atlas_esri_url_atlas) {
                  scope.esriConfig.urlAtlas = data.atlas_esri_url_atlas;
                  getAtlasEsriServiceInfo(scope.esriConfig.urlAtlas, 'atlas');
                  scope.esriConfig.atlas_line_models
                        = scope.esriConfig.atlas_area_models
                        = data.atlas_area_models;
                }
                else {
                  if (angular.isDefined(data['atlas_esri_url_line'])) {
                    scope.esriConfig.urlLine = data['atlas_esri_url_line'];
                    scope.esriConfig.atlas_line_models = data.atlas_line_models;

                    getAtlasEsriServiceInfo(scope.esriConfig.urlLine, 'line');
                  }

                  if (angular.isDefined(data['atlas_esri_url_area'])) {
                    scope.esriConfig.urlArea = data['atlas_esri_url_area'];
                    scope.esriConfig.atlas_area_models = data.atlas_area_models;

                    getAtlasEsriServiceInfo(scope.esriConfig.urlArea, 'area');
                  }
                }

                scope.atlasEsri.models = res.data.atlas_models;

                if (!data['atlas_esri_url_line']
                      && !data['atlas_esri_url_area']
                      && scope.atlas.mode === 'esri') {
                  getAtlasMxdModels();
                }
              }
            });

            console.info('atlas config uploaded');
          },
          function() {
            scope.atlas_models = [];
            console.error('no atlas config');
          }
        );


        var baseMapConfig, osmUrl;
        let getBaseMapConfig = () => {
          var basemaps = BaseMapFactory.resources.basemaps;
          for (var i = 0; i < basemaps.length; i++) {
            if (basemaps[i].active) {
              baseMapConfig = basemaps[i];
              osmUrl = baseMapConfig.url;
              break;
            }
          }
        };



        /**
         * Récupération de la liste des modèles d'atlas disponibles.
         * @param {*} urlAtlas URL de la requête HTTP du service d'atlas
         * @param {*} atlasResponse Réponse suite à l'appel du service d'atlas
         *                          en mode récupération de sa description
         */
        const initEsri108Atlas = (urlAtlas, atlasResponse) => {
          AtlasArcGIS108Service.initEsri108Atlas(scope, urlAtlas, atlasResponse);
          initEsriAtlasFormats(urlAtlas, 'area');
          if (!scope.atlasEsri.area_formats
              || scope.atlasEsri.area_formats.length === 0) {
            scope.atlasEsri.area_formats = ['pdf', 'jpeg'];
          }
          scope.atlasEsri.line_formats = scope.atlasEsri.area_formats;
        };


        /**
         *  Récupére les informations du service d'atlas esri
         * de type linéaire ou surfacique..
         *
         * @param {*} urlAtlas
         * @param {*} tempType : 'line' ou 'area'
         */
        let getAtlasEsriServiceInfo = (urlAtlas,tempType) => {

          if (angular.isDefined(urlAtlas) && urlAtlas.length > 0) {
            //TODO utiliser service de xgos
            $.get(urlAtlas + '?f=json').then(
              (info) => {
                scope.atlas[tempType + '_url_ok'] = true;
                if(typeof info == 'string')
                  parsedJsonAtlasResp = JSON.parse(info);
                else
                  parsedJsonAtlasResp = info;

                if (parsedJsonAtlasResp.error) {
                  require('toastr').error(
                    $filter('translate')('atlas.'+tempType+'&_wrong_configuration') +
                        parsedJsonAtlasResp.error.message
                  );
                  //-- Refaire l'affectation avec un délai
                  //-- pour bonne prise en compte par angularjs.
                  $timeout(() => {
                    scope.atlas[tempType + '_url_ok'] = false;
                  }, 250);
                }
                else {
                  if (tempType === 'atlas') {
                    initEsri108Atlas(urlAtlas, parsedJsonAtlasResp);
                  }
                  else {
                    initEsriAtlasTemplates(parsedJsonAtlasResp, tempType);
                    initEsriAtlasFormats(parsedJsonAtlasResp, tempType);
                  }
                  //-- Refaire l'affectation avec un délai
                  //-- pour bonne prise en compte par angularjs.
                  $timeout(() => { scope.atlas[tempType + '_url_ok'] = true; }, 250);
                }
              },
              (p1,p2,p3) => {
                scope.atlas[tempType + '_url_ok'] = false;
                scope.atlas[tempType + '_url_msg'] = p3;
                //-- Refaire l'affectation avec un délai
                //-- pour bonne prise en compte par angularjs.
                $timeout(() => { scope.atlas[tempType + '_url_ok'] = false; }, 250);
              }
            );
          }
        };



        /**
         * Défini la liste de modèle d'atlas à utiliser selon que
         *  l'on a choisi le type linéaire ou surfacique dans
         * l'Interface Homme Mahine.
         */
        let getAtlasMxdModels = () => {
          scope.atlas_models = scope.all_models
            .filter(model => model.type === 'arcgis');
          scope.atlasEsri.line_models = scope.atlas_models
            .filter(model => model.modelType === 'linear');
          scope.atlasEsri.area_models = scope.atlas_models
            .filter(model => model.modelType === 'surface')
            .map(model => model.name);
          scope.atlasEsri.area_formats = ['pdf', 'jpeg'];
          scope.atlasEsri.line_formats = ['pdf', 'jpeg'];
        };


        /**
         * Récupére la valeur d'un paramétre retourné dans le descriptif
         * du service de carte arcgis (linéaire ou surfacique).
         *
         * @param {*} serviceInfo
         * @param {*} name
         * @returns
         */
        let getParameterByName = (serviceInfo, name) => {
          if (
            !angular.isUndefined(serviceInfo.parameters) &&
              serviceInfo.parameters != null
          ) {
            for (var i = 0; i < serviceInfo.parameters.length; i++) {
              var loopParam = serviceInfo.parameters[i];

              if (loopParam.name == name) {
                return loopParam;
              }
            }
          }

          return null;
        };


        /**
         * Regarde dans la liste des modèles enregistrés si celui
         * dont le nom est en paramétre est sélectionné pour être utilisé ou non.
         *
         * @param {*} t: nom de modéle à chercher
         * @param {*} tempType: 'line' ou 'area'
         * @returns
         */
        let modelIsSelected = (t, tempType) => {
          let curLst = scope.esriConfig['atlas_' + tempType + '_models'];
          if (curLst) {
            for (let iTpl = 0; iTpl < curLst.length; iTpl++) {
              if (curLst[iTpl].name == t) {
                return curLst[iTpl].added;
              }
            }
          }
          //-- Nouveau modèle car absent de la liste,
          //-- donc, non sélectionné par défaut.
          return false;
        };


        //init Atlas line templates choice
        let initEsriAtlasTemplates = (serviceInfo, tempType) => {
          let templateParam = getParameterByName(serviceInfo, 'template');

          if (templateParam && angular.isDefined(templateParam.choiceList)) {
            scope.atlasEsri[tempType + '_models'] = templateParam.choiceList.map(
              (t) => {
                if (modelIsSelected(t, tempType)) {
                  return { name: t, added: true };
                }
                else {
                  return { name: t };
                }
              }
            );

            if (scope.atlasEsri[tempType + '_models'].length > 0) {
              if (angular.isDefined(templateParam.defaultValue)) {
                scope.atlasEsri[tempType + '_template'] =
                    templateParam.defaultValue;
              } else {
                scope.atlasEsri[tempType + '_template'] =
                    scope.atlasEsri[tempType + '_models'][0];
              }
            }
          }
        };


        //init Atlas line formats choice
        let initEsriAtlasFormats = (serviceInfo, tempType) => {
          let exportParam = getParameterByName(serviceInfo, 'export');

          if (exportParam && angular.isDefined(exportParam.choiceList)) {
            scope.atlasEsri[tempType+'_formats'] = exportParam.choiceList;

            if (scope.atlasEsri[tempType+'_formats'].length > 0) {
              if (angular.isDefined(exportParam.defaultValue)) {
                scope.atlasEsri[tempType + '_export_type'] =
                    exportParam.defaultValue;
              } else {
                scope.atlasEsri[tempType + '_export_type'] =
                    scope.atlasEsri[tempType+'_formats'][0];
              }
            }
          }
        };


        let saveConfig = () => {
          scope.atlas_models_validated = angular.copy(
            scope.atlas_models_configured
          );
          var count = scope.atlas_models_validated
            .map(function(x) {
              if (x.added) return x;
            })
            .filter(function(x) {
              if (x) return x;
            }).length;
          if (count === 0) {
            scope.atlas.atlas_model = undefined;
            initAtlas();
          }

          var datatosave = {
            //"atlas_models_validated" : scope.atlas_models_validated,
            atlas_models_configured: scope.atlas_models_configured,
            atlas_models: scope.atlas_models,
            atlas_esri_url_line: scope.esriConfig.urlLine,
            atlas_esri_url_area: scope.esriConfig.urlArea,
            atlas_esri_url_atlas: scope.esriConfig.urlAtlas,
            atlas_mode: scope.atlas.mode,
            atlas_line_models: scope.atlasEsri.line_models,
            atlas_area_models: scope.atlasEsri.area_models
          };
          ConfigFactory.add(datatosave, 'widgets', scope.ConfigName).then(
            function() {
              require('toastr').success(
                $filter('translate')('elastic.search.save_ok')
              );
            },
            function() {
              require('toastr').error(
                $filter('translate')('elastic.search.save_nok')
              );
            }
          );
        };


        var ngDialogPromise;
        scope.openConfig = function() {
          ngDialogPromise = ngDialog.openConfirm({
            template:
                'js/XG/widgets/mapapp/atlas/views/atlasConfiguration.html',
            scope: scope,
            className: 'ngdialog-theme-plain width800 nopadding miniclose',
          });
          if (!scope.atlas_models_configured)
            scope.atlas_models_configured = [];
          scope.configured_copy = angular.copy(
            scope.atlas_models_configured).filter(model => model.type !== 'arcgis');
          scope.atlas_models_copy = angular.copy(scope.atlas_models);
          ngDialogPromise.then(
            saveConfig,
            () => {
              scope.atlas_models_configured = angular.copy(
                scope.configured_copy
              );
              scope.atlas_models = angular.copy(scope.atlas_models_copy);
              require('toastr').error(
                $filter('translate')('elastic.search.annuler')
              );
            }
          );
        };


        /**
         * Ajout d'un modèle parmi ceux choisi pour le widget.
         *
         * @param {*} model
         * @param {*} templateType
         */
        scope.addToModel = function(model,templateType) {
          model.added = true;
          if (templateType == undefined) {
            scope.atlas_models_configured.push(model);
          }
        };


        /**
         * récupération de la liste des modèles selon le type:
         * linéaire ou surfacique.
         *
         * @param {*} templateType
         * @returns
         */
        let getTemplates = (templateType) => {
          let templates = scope.atlas_models;
          if (templateType) {
            templates = scope.atlasEsri[templateType + '_models'];
          }
          return templates;
        };


        scope.addAll = (templateType) => {
          let templates = getTemplates(templateType);
          templates.map((x) => {
            scope.addToModel(x,templateType);
          });
        };


        scope.removeFromModel = (model,templateType) => {
          model.added = false;
          if (templateType == undefined) {
            let idx = scope.atlas_models_configured
              .map(function (x) {
                return x.id;
              })
              .indexOf(model.id);
            scope.atlas_models_configured.splice(idx, 1);
          }
        };


        /**
         * Vidage de la liste des modèles choisis pour le widget
         * => aucun modèle choisi pour le widget.
         *
         * @param {*} templateType: 'line' ou 'area'
         */
        scope.initConfig = (templateType) => {
          if (templateType == undefined) {
            scope.atlas_models_configured = [];
          }
          let templates = getTemplates(templateType);
          templates.map(function(x) {
            x.added = false;
          });
        };


        let initAtlas = () => {
          scope.tiles = undefined;
          scope.features = undefined;
          scope.atlas.geometry = undefined;
          scope.atlas.active = false;
          try {
            if (!deregister)
              deregister = [
                scope.map.on('precompose', handlePreCompose),
                scope.map.on('postcompose', handlePostCompose),
                scope.map.getView().on('propertychange', function () {
                  if (scope.atlas.scale)
                    updatePrintRectanglePixels(scope.atlas.scale);
                }),
              ];
            deactivate();
          } catch (e) {
            console.log('no print rectangle');
          }
          scope.atlas.scale = undefined;
          if (scope.sommaire_layer) scope.map.removeLayer(scope.sommaire_layer);
        };


        let initAtlasScale = () => {
          scope.tiles = undefined;
          scope.features = undefined;
          scope.atlas.geometry = undefined;
          scope.atlas.active = false;
          try {
            if (!deregister)
              deregister = [
                scope.map.on('precompose', handlePreCompose),
                scope.map.on('postcompose', handlePostCompose),
                scope.map.getView().on('propertychange', function () {
                  if (scope.atlas.scale)
                    updatePrintRectanglePixels(scope.atlas.scale);
                }),
              ];
            deactivate();
          } catch (e) {
            console.log('no print rectangle');
          }
          //scope.atlas.scale = undefined;
          if (scope.sommaire_layer) {
            scope.map.removeLayer(scope.sommaire_layer);
            scope.sommaire_layer.getSource().clear();
          }
        };


        scope.$watch('atlas.atlas_model', function(newval) {
          if (!newval) return;
          initAtlas();
          if (!scope.capabilities) {
            return;
          }
          if (newval.printModel.length===0) {
            // -- Atlas sans modèle d'impression !!!
            require('toastr').error(
              $filter('translate')('atlas.noPrintTemplate'));
            return;
          }
          for (let i = 0; i < scope.capabilities.layouts.length; i++) {
            if (scope.capabilities.layouts[i].name === newval.printModel) {
              scope.atlas.print_model = scope.capabilities.layouts[i];
              scope.atlas.print.layout = newval.printModel;
              scope.atlas.print.activesommaire = newval.activesommaire;
              // -- Ticket 3372 --
              // -- Url d'accés à MapFishprint : même serveur que serveur KIS
              // -- avec redirection dans proxypass de Apache.
              scope.atlas.createURL = $location.protocol().concat('://',
                $location.host(), $location.port() ? ':' + $location.port() : '',
                '/printv2/pdf/create.json');
              scope.atlas.printURL = '/printv2/pdf/print.pdf';
              AtlasFactory.getAtlasModel(scope.atlas.print.app).then(
                function(res) {
                  if (res.data.etat === 'fini') {
                    scope.full_model = res.data.data;
                    scope.selected_model =
                          scope.full_model.layouts[newval.printModel].mainPage;
                    var size = AtlasService.calculateFullSize(
                      scope.selected_model
                    );
                    var items = AtlasService.createItems(size);
                    scope.atlas.sommaire = {
                      mainPage: angular.copy(scope.selected_model),
                    };
                    scope.atlas.sommaire.mainPage.items = items;
                    scope.atlas.sommaireName = newval.printModel + ' sommaire';
                    if (
                      Object.keys(scope.full_model.layouts).indexOf(
                        scope.atlas.sommaireName
                      ) === -1
                    ) {
                      scope.full_model.layouts[scope.atlas.sommaireName] =
                            scope.atlas.sommaire;
                      AtlasFactory.saveAtlasModel(
                        scope.atlas.print.app,
                        scope.full_model
                      ).then(
                        function() {
                          console.info('save atlas model success');
                        },
                        function() {
                          console.error('save atlas model failed');
                        }
                      );
                    } else {
                      scope.full_model.layouts[scope.atlas.sommaireName] =
                            scope.atlas.sommaire;
                    }
                  }
                },
                function() {
                  console.error('no model');
                }
              );
              break;
            }
          }
        });

        scope.$watch('atlas.scale', function(newval) {
          if (!newval) return;
          if (firstScale) {
            firstScale = false;
            scope.atlas.active = scope.atlas.active || false;
          }
          updatePrintRectanglePixels(newval);
          if (scope.atlas.geometry) createFullTiles(scope.atlas.geometry);
        });


        let updatePrintRectanglePixels = (scale) => {
          var value = Number(scale.value);
          printRectangle = calculatePageBoundsPixels(value);
          scope.map.render();
        };


        function calculatePageBoundsPixels(value) {
          var s = parseFloat(value);
          var size = scope.atlas.print_model.map; // papersize in dot!
          var view = scope.map.getView();
          var resolution = view.getResolution();
          var w =
              (((((size.width / POINTS_PER_INCH) * MM_PER_INCHES) / 1000.0) * s) /
                  resolution) *
              DEVICE_PIXEL_RATIO;
          var h =
              (((((size.height / POINTS_PER_INCH) * MM_PER_INCHES) / 1000.0) *
                  s) /
                  resolution) *
              DEVICE_PIXEL_RATIO;
          var mapSize = scope.map.getSize();
          var center = [
            (mapSize[0] * DEVICE_PIXEL_RATIO) / 2,
            (mapSize[1] * DEVICE_PIXEL_RATIO) / 2,
          ];

          var minx, miny, maxx, maxy;

          minx = center[0] - w / 2;
          miny = center[1] - h / 2;
          maxx = center[0] + w / 2;
          maxy = center[1] + h / 2;
          return [minx, miny, maxx, maxy];
        }

        // Compose events
        function handlePreCompose(evt) {
          var ctx = evt.context;
          ctx.save();
        }

        function handlePostCompose(evt) {
          var ctx = evt.context;
          var size = scope.map.getSize();
          var height = size[1] * DEVICE_PIXEL_RATIO;
          var width = size[0] * DEVICE_PIXEL_RATIO;

          var minx, miny, maxx, maxy;
          (minx = printRectangle[0]),
          (miny = printRectangle[1]),
          (maxx = printRectangle[2]),
          (maxy = printRectangle[3]);

          ctx.save();

          ctx.beginPath();
          // Outside polygon, must be clockwise
          ctx.moveTo(0, 0);
          ctx.lineTo(width, 0);
          ctx.lineTo(width, height);
          ctx.lineTo(0, height);
          ctx.lineTo(0, 0);
          ctx.closePath();

          // Inner polygon,must be counter-clockwise
          ctx.moveTo(minx, miny);
          ctx.lineTo(minx, maxy);
          ctx.lineTo(maxx, maxy);
          ctx.lineTo(maxx, miny);
          ctx.lineTo(minx, miny);
          ctx.closePath();

          ctx.fillStyle = 'rgba(0, 5, 25, 0.25)';
          ctx.fill();

          ctx.restore();
        }

        function getScaleFromResolution() {
          var resolution = scope.map.getView().getResolution();
          var dpi = 25.4 / 0.28;
          var mpu = scope.map
            .getView()
            .getProjection()
            .getMetersPerUnit();
          var inchesPerMeter = 39.37;
          return Math.ceil(
            parseFloat(resolution) * (mpu * inchesPerMeter * dpi)
          );
        }

        var activate = function() {
          deregister = [
            scope.map.on('precompose', handlePreCompose),
            scope.map.on('postcompose', handlePostCompose),
            scope.map.getView().on('propertychange', function () {
              if (scope.atlas.scale)
                updatePrintRectanglePixels(scope.atlas.scale);
            }),
          ];
          refreshComp();
        };

        var deactivate = function() {
          if (deregister) {
            try {
              for (var i = 0; i < deregister.length; i++) {
                ol.Observable.unByKey(deregister[i]);
                // deregister[i].target.unByKey(deregister[i]);
              }
            } catch (e) {
              console.info(e.message);
            }
          }
          refreshComp();
        };

        var refreshComp = function() {
          if (scope.atlas.scale) {
            updatePrintRectanglePixels(scope.atlas.scale);
            scope.map.render();
          }
        };

        scope.$watch('atlas.active', function(newVal) {
          if (newVal === true) {
            activate();
          } else {
            deactivate();
          }
        });

        function createTilesOnMap() {
          if (scope.sommaire_layer) scope.map.removeLayer(scope.sommaire_layer);

          var features = {
            type: 'FeatureCollection',
            totalFeatures: 0,
            crs: {
              type: 'name',
              properties: {
                name: 'EPSG:3857',
              },
            },
            features: [],
          };

          for (var i = 0; i < scope.tiles.extents.length; i++) {
            var f = {
              type: 'Feature',
              geometry: {
                type: 'Polygon',
                coordinates: [[]],
              },
              properties: {},
              id: i + 1,
            };
            var ext = scope.tiles.extents[i];
            var coordLL, coordLR, coordUR, coordUL;
            coordLL = [ext[0], ext[1]];
            coordLR = [ext[2], ext[1]];
            coordUR = [ext[2], ext[3]];
            coordUL = [ext[0], ext[3]];
            var coords = [coordLL, coordLR, coordUR, coordUL, coordLL];
            f.geometry.coordinates[0] = coords;
            features.features.push(f);
          }

          scope.atlas.features = features;
          scope.atlas.features.totalFeatures = features.features.length;

          var olf = new ol.format.GeoJSON().readFeatures(features);
          scope.features = olf;
          var originSourceDetail = new ol.source.Vector({
            name: 'sommaire',
          });
          originSourceDetail.addFeatures(olf);
          scope.sommaire_layer = new ol.layer.Vector({
            name: 'sommaire',
            source: originSourceDetail,
            style: (function() {
              var style = new ol.style.Style({
                fill: new ol.style.Fill({
                  color: 'rgba(0,0,0,0)',
                }),
                stroke: new ol.style.Stroke({
                  color: 'rgb(0,0,0)',
                  width: 2,
                  lineDash: [0.1, 5],
                }),
                text: new ol.style.Text({
                  font: '15px Calibri,sans-serif',
                  fill: new ol.style.Fill({
                    color: '#000',
                  }),
                  stroke: new ol.style.Stroke({
                    color: '#fff',
                    width: 3,
                  }),
                }),
              });
              var styles = [style];
              return function(feature) {
                style.getText().setText(feature.getId().toString());
                return styles;
              };
            })(),
          });
          scope.sommaire_layer.setZIndex(gaJsUtils.getZIndexMax(scope.map) + 1);
          ext = scope.sommaire_layer.getSource().getExtent();
          scope.center = [(ext[2] + ext[0]) / 2, (ext[3] + ext[1]) / 2];
          scope.atlas.print.pages = scope.tiles.pages;
          scope.showSommaireinMap = true;
          if (scope.atlas.active) scope.atlas.active = false;
          scope.toggleSommaire();
          scope.map.getView().fit(ext, scope.map.getSize());
          scope.fittedScale = getScaleFromResolution();
          scope.mapFittedSize = scope.map.getSize();
        }

        /**
         * Create the tiles for the Atlas
         * @param newVal: geometry of the selection
         */
        function createFullTiles(geometry) {
          scope.resetDrawGeometries();
          gaDomUtils.showGlobalLoader();
          var selection_extent;
          var print_extent = calculatePageBoundsPixels(scope.atlas.scale.value);
          if (scope.atlas.type === 'frame') {
            selection_extent = AtlasService.getPixelsFromExtent(geometry.getExtent());
          } else if (scope.atlas.type === 'line') {
            selection_extent = AtlasService.getPixelsFromCoordinates(geometry.getCoordinates());
          } else if (scope.atlas.type === 'polygon') {
            selection_extent = AtlasService.getPixelsFromExtent(geometry.getExtent());
          } else {
            //var ext = AtlasService.simplifyExtents(newVal);
            selection_extent = geometry.getExtent().map(function(x) {
              return AtlasService.getPixelsFromExtent(x);
            });
          }
          var layers = gclayers.getOperationalLayerCollection();

          if (scope.atlas.type === 'frame') {
            AtlasService.setTiles(
              layers,
              selection_extent,
              print_extent,
              scope.atlas
            ).then(
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              },
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              }
            );
          } else if (scope.atlas.type === 'line') {
            AtlasService.global(
              layers,
              selection_extent,
              print_extent,
              scope.atlas
            ).then(
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              },
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              }
            );
          } else if (scope.atlas.type === 'polygon') {
            // Use the extent of our polygon to copy the frame behaviour
            // Then filter them with the selectIntersectingTiles function
            AtlasService.setTiles(
              layers,
              selection_extent,
              print_extent,
              scope.atlas
            ).then(
              () => {
                scope.tiles = selectIntersectingTiles(geometry, AtlasService.getTiles());

                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              }
            );
          } else {
            AtlasService.setTilesFromSelection(
              layers,
              selection_extent,
              print_extent,
              scope.atlas
            ).then(
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              },
              function() {
                scope.tiles = AtlasService.getTiles();
                if (scope.tiles.extents.length === 0) initAtlasScale();
                else createTilesOnMap();

                gaDomUtils.hideGlobalLoader();
              }
            );
          }
        }

        /**
         * Return only the tiles intersecting with the geometry
         * @param geometry
         * @param tiles
         */
        const selectIntersectingTiles = (geometry, tiles) => {
          for(let i=tiles.extents.length-1; i>0; i--) {
            if(!geometry.intersectsExtent(tiles.extents[i])) {
              tiles.pixels.splice(i, 1);
              tiles.extents.splice(i, 1);
              tiles.distances.splice(i, 1);
              tiles.pages.splice(i, 1);
            }
          }
          return tiles;
        };


        scope.$watch('atlas.geometry', function(geometry) {
          if (geometry) {
            require('toastr').success(
              $filter('translate')('atlas.selectionsuccess')
            );
            try {
              createFullTiles(geometry);
            } catch (e) {
              console.error(e.message);
              atlas.geometry = undefined;
              gaDomUtils.hideGlobalLoader();
            }
          }
        });

        scope.toggleSommaire = function() {
          scope.showSommaireinMap
            ? scope.map.addLayer(scope.sommaire_layer)
            : scope.map.removeLayer(scope.sommaire_layer);
        };

        var toHexa = function(olColor) {
          var hex = '#';
          for (var i = 0; i < 3; i++) {
            var part = olColor[i].toString(16);
            if (part.length === 1 && parseInt(part) < 10) {
              hex += '0';
            }
            hex += part;
          }
          return hex;
        };


        /**
         * Affichage d'un SWeet ALert message pour indiquer que le job d'atlas a échoué.
         *
         * @param {*} scope Scope du widget à l'origine du (ou des) jobs.
         * @param {*} mess Message à afficher
         */
        const esriAtlasFailureMessage = (scope, mess) => {
          if (!scope.noJobMessage) {
            gaDomUtils.hideGlobalLoader();
            if (mess == undefined) mess = '';
            else mess = ': ' + mess;
            swal({
              title: $filter('translate')('common.print'),
              text: $filter('translate')('print.printFailure') + ' ' + mess,
              type: 'error',
              closeOnConfirm: false,
              closeOnCancel: false,
            });
          }
        };


        // Transform a ol.style.Style to a print literal object
        var transformToPrintLiteral = function(feature, style) {
          var literal = {
            zIndex: style.getZIndex(),
          };
          var fill = style.getFill();
          var stroke = style.getStroke();
          var textStyle = style.getText();
          var imageStyle = style.getImage();

          if (imageStyle) {
            var size = imageStyle.getSize();
            var anchor = imageStyle.getAnchor();
            var scale = imageStyle.getScale();
            literal.rotation = imageStyle.getRotation();
            if (size) {
              literal.graphicWidth = size[0] * scale;
              literal.graphicHeight = size[1] * scale;
            }
            if (anchor) {
              literal.graphicXOffset = -anchor[0] * scale;
              literal.graphicYOffset = -anchor[1] * scale;
            }
            if (imageStyle instanceof ol.style.Icon) {
              literal.externalGraphic = imageStyle.getSrc();
              literal.fillOpacity = 1;
            } else {
              // ol.style.Circle
              fill = imageStyle.getFill();
              stroke = imageStyle.getStroke();
              literal.pointRadius = imageStyle.getRadius();
            }
          }

          if (fill) {
            var color = ol.color.asArray(fill.getColor());
            literal.fillColor = toHexa(color);
            literal.fillOpacity = color[3];
          } else if (!literal.fillOpacity) {
            literal.fillOpacity = 0; // No fill
          }

          if (stroke) {
            color = ol.color.asArray(stroke.getColor());
            literal.strokeWidth = stroke.getWidth();
            literal.strokeColor = toHexa(color);
            literal.strokeOpacity = color[3];
            literal.strokeLinecap = stroke.getLineCap() || 'round';
            literal.strokeLinejoin = stroke.getLineJoin() || 'round';

            if (stroke.getLineDash()) {
              literal.strokeDashstyle = 'dash';
            }
            // TO FIX: Not managed by the print server
            // literal.strokeMiterlimit = stroke.getMiterLimit();
          } else {
            literal.strokeOpacity = 0; // No Stroke
          }

          if (textStyle) {
            var fillColor = ol.color.asArray(textStyle.getFill().getColor());
            var strokeColor = ol.color.asArray(
              textStyle.getStroke().getColor()
            );
            var fontValues = textStyle.getFont().split(' ');
            literal.fontColor = toHexa(fillColor);
            // Fonts managed by print server: COURIER, HELVETICA, TIMES_ROMAN
            literal.fontFamily = fontValues[1].toUpperCase();
            literal.fontSize = parseInt(fontValues[0].replace('px', ''));
            //literal.fontWeight = fontValues[0];
            literal.label = textStyle.getText();
            literal.labelAlign = textStyle.getTextAlign();
            literal.labelOutlineColor = toHexa(strokeColor);
            literal.labelOutlineWidth = textStyle.getStroke().getWidth();
          }

          return literal;
        };

        scope.encoders = {
          layers: {
            Layer: function(layer) {
              var enc = {
                layer: layer.bodId,
                opacity: layer.getOpacity(),
              };
              return enc;
            },
            Group: function(layer, proj) {
              var encs = [];
              var subLayers = layer.getLayers();
              subLayers.forEach(function(subLayer) {
                if (subLayer.visible) {
                  var enc = scope.encoders.layers['Layer'].call(this, layer);
                  var layerEnc = encodeLayer(subLayer, proj);
                  if (layerEnc && layerEnc.layer) {
                    $.extend(enc, layerEnc);
                    encs.push(enc.layer);
                  }
                }
              });
              return encs;
            },
            Vector: function(layer, features) {
              var enc = scope.encoders.layers['Layer'].call(this, layer);
              var format = new ol.format.GeoJSON();
              var encStyles = {};
              var encFeatures = [];
              var styleId = 0;
              var hasLayerStyleFunction = !!(
                layer.getStyleFunction && layer.getStyleFunction()
              );

              angular.forEach(features, function(feature) {
                var encStyle = {
                  id: styleId,
                };
                var styles = hasLayerStyleFunction
                  ? layer.getStyleFunction()(feature)
                  : ol.feature.defaultStyleFunction(feature);

                var geometry = feature.getGeometry();

                // Transform an ol.geom.Circle to a ol.geom.Polygon
                if (geometry.getType() === 'Circle') {
                  var polygon = pv2LayerServices.circleToPolygon(geometry);
                  feature = new ol.Feature(polygon);
                }

                var encJSON = JSON.parse(format.writeFeature(feature));
                if (!encJSON.properties) {
                  encJSON.properties = {};

                  // Fix https://github.com/geoadmin/mf-geoadmin3/issues/1213
                } else if (encJSON.properties.Style) {
                  delete encJSON.properties.Style;
                }

                encJSON.properties._gx_style = styleId;
                encFeatures.push(encJSON);

                if (styles && styles.length > 0) {
                  $.extend(
                    encStyle,
                    transformToPrintLiteral(feature, styles[0])
                  );
                }

                encStyles[styleId] = encStyle;
                styleId++;
              });
              angular.extend(enc, {
                type: 'Vector',
                styles: encStyles,
                styleProperty: '_gx_style',
                geoJson: {
                  type: 'FeatureCollection',
                  features: encFeatures,
                },
                name: layer.bodId,
                opacity: layer.opacity != null ? layer.opacity : 1.0,
              });
              return enc;
            },
            WMS: function(layer, config) {
              var enc = scope.encoders.layers['Layer'].call(this, layer);
              var params = layer.getSource().getParams();

              //
              let layers = params.LAYERS;
              if (!layers) {
                layers = params.layers;
              }
              if (!layers) {
                gaDomUtils.hideGlobalLoader();
                esriAtlasFailureMessage($filter('translate')(
                  'tools.atlas.no_layer'
                ));
                return;
              }
              layers = layers.split(',') || [];
              var styles =
                  params.STYLES !== undefined
                    ? params.STYLES.split(',')
                    : new Array(layers.length).join(',').split(',');
              var urlw =
                $location.protocol() +
                '://' +
                $location.host() +
                ':' +
                $location.port() +
                '/services/' +
                PortalsFactory.getPortalId() +
                '/geoserver/wms?token=' +
                localStorage.auth_token;

              if (layer.type == 'GEOWEBCACHE') {
                //console.log(layer.getSource().getUrls());

                angular.extend(enc, {
                  type: 'tms',
                  baseURL: layer
                    .getSource()
                    .getUrls()[0]
                    .replace('wms', 'tms'),
                  layer: layers[0],
                  //styles: styles,
                  format: 'png',
                  resolutions: [
                    156543.0339,
                    78271.51695,
                    39135.758475,
                    19567.8792375,
                    9783.93961875,
                    4891.969809375,
                    2445.9849046875,
                    1222.99245234375,
                    611.496226171875,
                    305.7481130859375,
                    152.87405654296876,
                    76.43702827148438,
                    38.21851413574219,
                    19.109257067871095,
                    9.554628533935547,
                    4.777314266967774,
                    2.388657133483887,
                    1.1943285667419434,
                    0.5971642833709717,
                  ],
                  maxExtent: [
                    -20037508.3392,
                    -20037508.3392,
                    20037508.3392,
                    20037508.3392,
                  ],
                  tileSize: [256, 256],

                  singleTile: false,
                });
                return enc;
              } else if (layer.type == 'OSM') {
                if (osmUrl) {
                  angular.extend(enc, {
                    type: 'OSM',
                    baseURL: osmUrl,
                    singleTile: false,
                    maxExtent: [
                      -20037508.3392,
                      -20037508.3392,
                      20037508.3392,
                      20037508.3392,
                    ],
                    tileSize: [256, 256],
                    extension: 'png',
                    resolutions: [
                      156543.0339,
                      78271.51695,
                      39135.758475,
                      19567.8792375,
                      9783.93961875,
                      4891.969809375,
                      2445.9849046875,
                      1222.99245234375,
                      611.496226171875,
                      305.7481130859375,
                      152.87405654296876,
                      76.43702827148438,
                      38.21851413574219,
                      19.109257067871095,
                      9.554628533935547,
                      4.777314266967774,
                      2.388657133483887,
                      1.1943285667419434,
                      0.5971642833709717,
                    ],
                  });
                  return enc;
                }
              } else if (layer.type == 'WMSBACK') {
                angular.extend(enc, {
                  type: 'WMS',
                  baseURL: layer.getSource().getUrls()[0],
                  layers: layer
                    .getSource()
                    .getParams()
                    .LAYERS.split(','),
                  version: layer.getSource().getParams().VERSION || '1.1.1',
                  styles: layer.getSource().getParams().STYLES,
                  format: 'image/jpeg', // + (config.format || 'png'),
                  customParams: {
                    EXCEPTIONS: 'XML',
                    TRANSPARENT: 'true',
                    CRS: scope.map
                      .getView()
                      .getProjection()
                      .getCode(),
                    TIME: params.TIME,
                  },
                  singleTile: false,
                });
                return enc;
              } else {
                //console.log(layer.actualEventTarget_);
                angular.extend(enc, {
                  type: 'WMS',
                  baseURL: config.wmsUrl || urlw,
                  layers: layers,
                  styles: styles,
                  format: 'image/' + (config.format || 'png'),
                  customParams: {
                    EXCEPTIONS: 'XML',
                    TRANSPARENT: 'true',
                    TIME: params.TIME,
                  },
                  singleTile: config.singleTile || false,
                });
                return enc;
              }
            },
            OSM: function(layer) {
              if (osmUrl) {
                var enc = scope.encoders.layers['Layer'].call(this, layer);

                angular.extend(enc, {
                  type: 'OSM',
                  baseURL: osmUrl,
                  singleTile: false,
                  maxExtent: [
                    -20037508.3392,
                    -20037508.3392,
                    20037508.3392,
                    20037508.3392,
                  ],
                  tileSize: [256, 256],
                  extension: 'png',
                  resolutions: [
                    156543.0339,
                    78271.51695,
                    39135.758475,
                    19567.8792375,
                    9783.93961875,
                    4891.969809375,
                    2445.9849046875,
                    1222.99245234375,
                    611.496226171875,
                    305.7481130859375,
                    152.87405654296876,
                    76.43702827148438,
                    38.21851413574219,
                    19.109257067871095,
                    9.554628533935547,
                    4.777314266967774,
                    2.388657133483887,
                    1.1943285667419434,
                    0.5971642833709717,
                  ],
                });
                return enc;
              }
            },
            WMTS: function(layer) {
              var enc = scope.encoders.layers['Layer'].call(this, layer);
              var source = layer.getSource();
              var tileGrid = source.getTileGrid();

              /*var matrixIds = [];
                            for ( var i = 0 ; i < tileGrid.getMatrixIds().length ; i++){
                                matrixIds.push({
                                    "identifier": tileGrid.getMatrixIds()[i],
                                    "matrixSize": [ 1 , 1 ] ,
                                    "resolution": tileGrid.getResolutions()[i],
                                    "tileSize": [ tileGrid.getTileSize(), tileGrid.getTileSize()],
                                    "topLeftCorner": [ -20037508.3392 , 20037508.3392 ]
                                });
                            }*/

              angular.extend(enc, {
                type: 'WMTS',
                baseURL: source.getUrls()[0],
                opacity: layer.getOpacity(),
                layer: source.getLayer(),
                maxExtent: source.getProjection().getExtent(),
                tileOrigin: ol.extent.getTopLeft(
                  source.getProjection().getExtent()
                ),
                tileSize: [256, 256],
                resolutions: tileGrid.getResolutions(),
                zoomOffset: tileGrid.getMinZoom(),
                version: source.getVersion(),
                requestEncoding: 'KVP',
                extension: source.getFormat().split('/')[1],
                //matrixIds : matrixIds,
                format: source.getFormat(),
                //style: 'default',
                /*dimensions: [ 'TIME' ],
                                params: { 'TIME': source.getDimensions().Time },*/
                matrixSet: source.getMatrixSet(),
              });

              return enc;
            },
          },
          legends: {
            ga_urllegend: function(layer, config) {
              var format = '.png';
              if (scope.options.pdfLegendList.indexOf(layer.bodId) != -1) {
                format = pdfLegendString;
              }
              var enc = scope.encoders.legends.base.call(this, config);
              enc.classes.push({
                name: '',
                icon:
                    scope.options.legendUrl +
                    layer.bodId +
                    '_' +
                    $translate.uses() +
                    format,
              });
              return enc;
            },
            base: function(config) {
              return {
                name: config.label,
                classes: [],
              };
            },
          },
        };

        let canPrintLayer = (layer, featuresList, resolution, addedLayer) => {
          let fti = layer.get('fti');
          if (fti && fti.type == 'esri') {
            return false;
          }
          return layer.visible &&
              (layer.getMinResolution() == undefined ||
                  resolution >= layer.getMinResolution()) &&
              (layer.getMaxResolution() == Infinity ||
                  resolution < layer.getMaxResolution()) &&
              layer.name !== 'sommaire' &&
              (!fti || featuresList.length === 0 ||
                  ( fti &&
                      featuresList.indexOf(fti.name) !== -1 &&
                      addedLayer.indexOf(fti.name) === -1));
        };

        /**
         *  Création de la légende de l'atlas dans le cas ou l'on veut
         *  les couches visible sur la carte
         */
        const setLayersToPrint = (layers, proj, encLayers, outParams) => {
          const functionDefer = $q.defer();
          const resolution = scope.map.getView().getResolution();

          // Lancement des promesses de récupération de la liste des features présentes
          // pour chaque tuile de notre atlas
          const getFeaturesListPromises = [];
          for (let extentp of scope.tiles.extents) {
            // Selon que l'on veut l'impression des features visible
            // sur la carte ou dans la configuration
            if (scope.atlas.visibleonscreen === 'asVisibleOnMap') {
              getFeaturesListPromises.push(AtlasService.getfeaturesListInExtent(layers, extentp));
            } else {
              getFeaturesListPromises.push(
                AtlasService.getfeaturesListInExtentForConfig(layers, extentp, scope.atlas));
            }
          }

          // Lancement des promesses de récupération des légendes pour chaque tuile de l'atlas
          $q.all(getFeaturesListPromises).then(featuresListsRes => {
            const getLegendForLayersPromises = [];

            // Itération sur les réponses (correspond à une itération sur chaque tuile)
            for (let i=0; i<featuresListsRes.length; i++) {

              const featuresList = $.unique(featuresListsRes[i].data.features.map(
                x => { return x.id.split('.')[0]; })
              );
              const addedLayer = [];
              let LayersinExtent = [];
              const ftisAndStyles = { ftiAndStyle: [] };
              outParams.encLegends = outParams.encLegends || [];
              angular.forEach(layers, layer => {
                if (canPrintLayer(layer,featuresList,resolution,addedLayer)) {
                  addedLayer.push(layer.name);
                  if (layer instanceof ol.layer.Group) {
                    let encs = scope.encoders.layers['Group'].call(this, layer, proj);
                    LayersinExtent = LayersinExtent.concat(encs);
                  } else {
                    let enc = encodeLayer(layer, proj);
                    if (enc && enc.layer) {
                      LayersinExtent.push(enc.layer);
                    }
                  }
                  // Ajout des fti et styles pour la légende à afficher dans la tuile
                  if (layer.fti) {
                    ftisAndStyles.ftiAndStyle.push({
                      ftiUid: layer.fti.uid,
                      style: layer.getSource().getParams().STYLES,
                      name: layer.name
                    });
                  }
                }
              });
              encLayers.push(LayersinExtent);

              // Récupération de la configuration de la légende
              let extentp = scope.tiles.extents[i];

              getLegendForLayersPromises.push(
                PrintLegendFactory.getLegendForLayers(extentp[0], extentp[2],
                  extentp[3], extentp[1],
                  scope.map.getView().getProjection().getCode(),
                  parseInt('' + scope.atlas.scale.value), ftisAndStyles));
            }

            // Completion des infos de légende et retour de la promesse
            $q.all(getLegendForLayersPromises).then(legendsDataRes => {
              // Itération sur les réponses (correspond à une itération sur chaque tuile)
              for (let legendsData of legendsDataRes) {
                let encLegend = [];
                layers.forEach((layer) => {
                  for (let legendData of legendsData.data) {
                    if (layer.fti && layer.fti.uid === legendData.fti) {
                      encLegend.push({
                        classes: legendData.classes,
                        theme: layer.fti.theme,
                        name: layer.fti.name
                      });
                    }
                  }
                });
                outParams.encLegends.push(encLegend);
              }
            }).finally(() => functionDefer.resolve());
          });

          return functionDefer.promise;
        };


        /**
         * Calcule l'étendue de toutes les pages de l'atlas.
         *
         * @return {Array<number>} Un tableau représentant l'étendue
         *  de toutes les tuiles au format [minX, minY, maxX, maxY].
         */
        const getExtentOfTiles = () => {
          let minX = Infinity, minY = Infinity,
            maxX = -Infinity, maxY = -Infinity;
          for (const tileExtent of scope.tiles.extents) {
            minX = Math.min(minX, tileExtent[0]);
            minY = Math.min(minY, tileExtent[1]);
            maxX = Math.max(maxX, tileExtent[2]);
            maxY = Math.max(maxY, tileExtent[3]);
          }
          return [minX, minY, maxX, maxY];
        };


        // Encode ol.Layer to a basic js object
        const encodeLayer = (layer, proj) => {
          var encLayer, encLegend;
          var resolution = scope.map.getView().getResolution();
          let ext = proj.getExtent();
          if (ext===null) {
            ext = getExtentOfTiles();
          }

          if (!(layer instanceof ol.layer.Group) && layer.getSource != undefined) {
            let src = layer.getSource();
            var layerConfig = /*gaLayers.getLayer(layer.bodId) ||*/ {};
            var minResolution = layerConfig.minResolution || 0;
            var maxResolution = layerConfig.maxResolution || Infinity;

            if (resolution <= maxResolution && resolution >= minResolution) {
              if (src instanceof ol.source.WMTS) {
                encLayer = scope.encoders.layers['WMTS'].call(
                  this,
                  layer,
                  layerConfig
                );
              } else if (
                src instanceof ol.source.ImageWMS ||
                  src instanceof ol.source.TileWMS
              ) {
                encLayer = scope.encoders.layers['WMS'].call(
                  this,
                  layer,
                  layerConfig
                );
              } else if (src instanceof ol.source.OSM) {
                encLayer = scope.encoders.layers['OSM'].call(
                  this,
                  layer,
                  layerConfig
                );
              } else if (
                src instanceof ol.source.Vector ||
                  src instanceof ol.source.ImageVector
              ) {
                if (src instanceof ol.source.ImageVector) {
                  src = src.getSource();
                }

                var features = [];
                if (src.getFeatures().length!==0) {
                  src.forEachFeatureInExtent(ext, function(feat) {
                    features.push(feat);
                  });
                }

                if (features && features.length > 0) {
                  encLayer = scope.encoders.layers['Vector'].call(
                    this,
                    layer,
                    features
                  );
                }
              }
            }
          }

          if (scope.atlas.print.legend /*&& layerConfig.hasLegend*/) {
            if (
              layer.visible &&
                layer.name.toUpperCase() != 'OSM' &&
                layer.type != 'OSM' &&
                layer.type != 'GEOWEBCACHE' &&
                layer.type != 'WMSBACK'
            ) {
              var l = '';

              l =
              'http://localhost' +
              //
              //$location.protocol() +
              //'://' +
              //$location.host() +
              //':' +
              //$location.port() +
              '/services/' +
              PortalsFactory.getPortalId() +
              '/geoserver/' +
              '/wms?SERVICE=WMS&REQUEST=GetLegendGraphic' +
              '&VERSION=1.0.0&FORMAT=image/png&WIDTH=20&' +
              'HEIGHT=20&LAYER=' +
              layer.name +
              '&token=' +
              localStorage.getItem('auth_token');
              if (!angular.isUndefined(layer.style)) {
                l += '&STYLE=' + layer.style;
              }
              encLegend = {
                classes: [
                  {
                    icons: [l],
                    name: '',
                    iconBeforeName: true,
                    typeInfo: ''
                  },
                ],
                name:
                    $filter('translate')(
                      'features.' + layer.name + '.alias'
                    ).indexOf('features.') === -1
                      ? $filter('translate')('features.' + layer.name + '.alias')
                      : layer.label,
              };
            }
          }

          return { layer: encLayer, legend: encLegend };
        };

        /**
         * Fonction d'appel de l'impression d'atlas depuis l'extérieur de la directive
         */
        scope.printAtlasCall = () => {
          // Impression normale
          if (!atlas.mode.startsWith('esri')) {
            scope.printAtlas();
          }
          // Impression esri
          else {
            scope.printAtlasEsri();
          }
        };


        scope.printAtlas = function() {
          gaDomUtils.showGlobalLoader();
          try {
            getBaseMapConfig();
            var view = scope.map.getView();
            var proj = view.getProjection();
            var encLayers = [];
            var layers = gclayers.getOperationalLayerCollection();
            var backl = scope.map.getLayers();
            var booladdedbackground = false;
            backl.forEach(function(item) {
              if (item.type == 'OSM') {
                layers.insertAt(0, item);
                booladdedbackground = true;
              }
              if (item.type == 'GEOWEBCACHE') {
                layers.insertAt(0, item);
                booladdedbackground = true;
              }
              if (item.type == 'WMSBACK') {
                layers.insertAt(0, item);
                booladdedbackground = true;
              }
              if (item.type == 'WMTS') {
                layers.insertAt(0, item);
              }
            });
            var outParams = {};

            scope.atlas.sommairePrint = angular.copy(scope.atlas.print);

            setLayersToPrint(layers, proj, encLayers, outParams).then(() => {
              var overlays = scope.map.getOverlays();

              overlays.forEach(function(overlay) {
                var center = overlay.getPosition();
                if (center) {
                  var marker = {
                    type: 'Vector',
                    styles: {
                      '1': {
                        externalGraphic: scope.markerUrl,
                        graphicWidth: 20,
                        graphicHeight: 30,
                        // the icon is not perfectly centered in the image
                        // these values must be the same in map.less
                        graphicXOffset: -12,
                        graphicYOffset: -30,
                      },
                    },
                    styleProperty: '_gx_style',
                    geoJson: {
                      type: 'FeatureCollection',
                      features: [
                        {
                          type: 'Feature',
                          properties: {
                            _gx_style: 1,
                          },
                          geometry: {
                            type: 'Point',
                            coordinates: [center[0], center[1], 0],
                          },
                        },
                      ],
                    },
                    name: 'drawing',
                    opacity: 1,
                  };
                  encLayers.push(marker);
                }
              });

              angular.extend(scope.atlas.print, {
                layers: angular.copy(encLayers),
                legends: angular.copy(outParams.encLegends),
              });

              var resultatSommaire = AtlasService.getUniqueLayers(
                encLayers,
                outParams.encLegends,
                scope.atlas.print.pages,
                scope.fittedScale,
                scope.mapFittedSize,
                scope.atlas.sommaire.mainPage.items
              );
              resultatSommaire.pages[0].scale *= 1.2;
              resultatSommaire.pages[0].scale = Math.ceil(
                resultatSommaire.pages[0].scale
              );

              angular.extend(scope.atlas.sommairePrint, {
                layers: angular.copy(resultatSommaire.layers),
                legends: angular.copy(resultatSommaire.legends),
              });

              resultatSommaire.pages[0].center = scope.center;
              scope.atlas.sommairePrint.pages = angular.copy(
                resultatSommaire.pages
              );

              var vectors = scope.encoders.layers['Vector'].call(
                this,
                scope.sommaire_layer,
                scope.features
              );

              scope.atlas.sommairePrint.layers.push(vectors);
              scope.atlas.sommairePrint.layout = scope.atlas.sommaireName;
              scope.full_model.scales.push(resultatSommaire.pages[0].scale);

              if (booladdedbackground) layers.removeAt(0);

              AtlasFactory.saveAtlasModel(
                scope.atlas.print.app,
                scope.full_model
              ).then(
                function() {
                  // Quand on garde la géométrie dans le paramètre dans print(),
                  // On a une erreur cyclic value pour notre requete
                  // Cette géométrie n'est pas utilisée ensuite, donc je l'enlève
                  const noCyclicAtlas = angular.copy(scope.atlas);
                  noCyclicAtlas.geometry = undefined;
                  if (attrs.ongeturl) {
                    // -- Exemple: fonction loadUrlToBytesAuto de dictWidget
                    // -- qui appelle downloadPrintUrl
                    // -- pour laquelle on fourni les paramétres ci-dessous
                    noCyclicAtlas.atlasFileName = 'plan_' + attrs.dictfilename;
                  }
                  AtlasFactory.print(noCyclicAtlas).then(
                    function(res) {
                      var stop = $interval(function() {
                        AtlasFactory.getProgression(res.data.uid).then(
                          function(res) {
                            if (res.data.progression === 100) {
                              $interval.cancel(stop);
                              responseGetIt(res.data);
                            }
                          },
                          function() {
                            gaDomUtils.hideGlobalLoader();
                            $interval.cancel(stop);
                            require('toastr').error(
                              $filter('translate')('atlas.printechech')
                            );
                          }
                        );
                      }, 3000);
                    },
                    function() {
                      require('toastr').error(
                        $filter('translate')('atlas.printechech')
                      );
                      gaDomUtils.hideGlobalLoader();
                    }
                  );
                },
                function() {
                  require('toastr').error(
                    $filter('translate')('atlas.printechech')
                  );
                  gaDomUtils.hideGlobalLoader();
                }
              );
            });
          } catch (e) {
            console.error(e.message);
            gaDomUtils.hideGlobalLoader();
            require('toastr').error($filter('translate')('atlas.printechech'));
          }
        };


        scope.resetDrawGeometries = function() {
          clearDrawing();
        };


        function clearDrawing() {
          scope.atlasEsri.areaGeometry = undefined;
          scope.atlasEsri.lineGeometry = undefined;

          try {
            let drawSource = gclayers.getDrawLayer().getSource();
            if (drawSource != undefined) {
              drawSource.clear();
            }
          } catch (e) {
            console.info(e.message);
          }

          scope.SelectionLayer.setVisible(false);
          if (scope.sommaire_layer) {
            scope.map.removeLayer(scope.sommaire_layer);
          }

          // Supprime les sélections en cours dans atlasSelection
          scope.$broadcast('removeSelection');
        }


        scope.$on('atlas_clearDrawing', () => {
          clearDrawing();
        });


        scope.$watch('atlas.mode', (newVal) => {
          scope.atlasMode = scope.atlas.mode;
          if (newVal === 'esri' && !scope.esriConfig.urlLine && !scope.esriConfig.urlArea) {
            getAtlasMxdModels();
          } else if (scope.all_models) {
            scope.atlas_models = scope.all_models;
            if (scope.atlas_models.length > 0) {
              scope.atlas_models_validated = [];
            }
          }
        });


        let loadDataStores = () => {
          DataStoreFactory.get().then(function(res) {
            if (res.data.length != 0) {
              for (var indice in res.data) {
                scope.datasources.push(res.data[indice]);
                if (scope.esriConfig.datasource
                    && scope.esriConfig.datasource.name == res.data[indice].name) {
                  scope.esriConfig.datasource = res.data[indice];
                }
              }
              if (res.data.length == 1) {
                scope.esriConfig.datasource = res.data[0];
              }
            }
          });
        };
        scope.datasources = [];


        /**
         * Lancement de la construction de l'atlas ArcGIS
         * suite à clic utilisateur sur commande bleue (imprimante).
         */
        scope.printAtlasEsri = () => {
          if (attrs.dictid && attrs.dictftiname) {
            scope.atlasEsri.queryDefinitions
              = `[{"layerName":"${attrs.dictftiname}", "queryDefinition": "id = ${attrs.dictid}"}]`;
          }
          const params = {
            atlasEsri: scope.atlasEsri,
            esriConfig: scope.esriConfig,
            atlasMode: scope.atlasMode,
            scope,
            atlasResponse: parsedJsonAtlasResp
          };
          if (attrs.ongeturl) {
            // -- Exemple: fonction loadUrlToBytesAuto de dictWidget qui appelle downloadPrintUrl
            // -- pour laquelle on fourni les paramétres ci-dessous
            params.atlasFunction = scope[attrs.ongeturl.replace('()','')];
            params.atlasFunctionParams = {};
            params.atlasFunctionParams.docId = attrs.dictid;
            params.atlasFunctionParams.atlasFileName = attrs.dictfilename;
          }
          AtlasService.submitAtlasEsri(params);
        };


        function generateErrorMsg(errors) {
          var errorMsg = $filter('translate')('atlas.standarterror');
          var err = [];
          errors.map(function(x) {
            switch (x.text) {
              case 'atlas.errorpagedegarde':
                err.push($filter('translate')('atlas.errorpagedegarde'));
                break;
              case 'atlas.errorpagedefin':
                err.push($filter('translate')('atlas.errorpagedefin'));
                break;
              case 'atlas.errorsommaire':
                err.push($filter('translate')('atlas.errorsommaire'));
                break;
              case 'atlas.errorpage':
                err.push(
                  $filter('translate')('atlas.errorpage') + ' ' + (x.page + 1)
                );
                break;
            }
          });
          errorMsg += err.join(',');
          require('toastr').error(errorMsg);
        }


        const responseGetIt = (data) => {

          gaDomUtils.hideGlobalLoader();

          if (data.response && data.response.errors && data.response.errors.length > 0)
            generateErrorMsg(data.response.errors);

          if (attrs.ongeturl) {
            scope[attrs.res].value = data.response.url;
            scope[attrs.ongeturl.replace('()', '')]();
            $timeout(clearDrawing, 1000);
          } else {
            window.open(data.response.url);
          }
        };


        scope.$on('openTools_atlas', function(event, arg) {
          if (arg.directive === 'atlas') {
            if (angular.isDefined(scope.atlas_models)) {
              if (
                scope.atlas_models.length === 0 &&
                !scope.atlas.mode.startsWith('esri')
              ) {
                require('toastr').info($filter('translate')('atlas.nomodel'));
              }
            }
            if (scope.showSommaireinMap && scope.sommaire_layer)
              scope.map.addLayer(scope.sommaire_layer);
          }
        });

        scope.$on('closeTools_atlas', function(event, arg) {
          if (arg.directive === 'atlas') {
            clearDrawing();
            if (scope.atlas.active) {
              scope.atlas.active = false;
              try {
                if (!deregister)
                  deregister = [
                    scope.map.on('precompose', handlePreCompose),
                    scope.map.on('postcompose', handlePostCompose),
                    scope.map.getView().on('propertychange', function() {
                      if (scope.atlas.scale)
                        updatePrintRectanglePixels(scope.atlas.scale);
                    }),
                  ];
                deactivate();
              } catch (e) {
                console.log('no print rectangle');
              }
            }
          }
        });


        scope.$on('closingCategory', function () {
          scope.resetDrawGeometries();
        });


        scope.$on('switchingCategory', function () {
          scope.resetDrawGeometries();
        });


        scope.$on('openCloseTools_atlas', function(event, arg) {
          if (arg.directive === 'atlas') {
            if (arg.active) {
              scope.SelectionLayer.setVisible(true);
              if (scope.atlas_models.length === 0) {
                require('toastr').info($filter('translate')('atlas.nomodel'));
              }

              if (scope.showSommaireinMap && scope.sommaire_layer)
                scope.map.addLayer(scope.sommaire_layer);
            }
            if (scope.atlas.active) {
              scope.atlas.active = false;
              try {
                if (!deregister)
                  deregister = [
                    scope.map.on('precompose', handlePreCompose),
                    scope.map.on('postcompose', handlePostCompose),
                    scope.map.getView().on('propertychange', function() {
                      if (scope.atlas.scale)
                        updatePrintRectanglePixels(scope.atlas.scale);
                    }),
                  ];
                deactivate();
              } catch (e) {
                console.log('no print rectangle');
              }
            }
            scope.resetDrawGeometries();
          }
        });


        let updateAtlasEsri = (tempType) => {
          let now = new Date();
          console.log('updateAtlasEsri scope.lastUpdate='+scope.lastUpdate);
          console.log('updateAtlasEsri  now='+now);
          console.log('updateAtlasEsri  now - scope.lastUpdate='+(now - scope.lastUpdate));
          if (now - scope.lastUpdate < 1000) {
            $timeout(() => { updateAtlasEsri(tempType); }, 500);
          }
          else {
            scope.lastUpdate = undefined;
            getAtlasEsriServiceInfo(scope.atlasUrlInput[tempType], tempType);
          }
        };


        let updateAtlasEsri0 = (tempType, url) => {
          //-- Mettre à jour la liste des modèles d'atlas.
          //-- Si pas de modification de l'url en cours,
          //-- on lance l'attente de fin de modification.
          if (url) {
            scope.atlasUrlInput[tempType] = url;
          }
          let now = new Date();
          console.log('scope.lastUpdate=' + scope.lastUpdate);
          console.log('now - scope.lastUpdate=' + now + ' - '
              + scope.lastUpdate + ' = ' + (now - scope.lastUpdate));
          console.log('-->');
          if (scope.lastUpdate == undefined || now - scope.lastUpdate > 5000) {
            scope.lastUpdate = now;
            updateAtlasEsri(tempType);
          }
          else {
            //-- Date de dernière modification de l'URL.
            scope.lastUpdate = now;
          }
        };


        scope.sceTabs = {};
        scope.sceTabs.tabs = [{ label: 'tools.atlas.linear_atlases' },
          { label: 'tools.atlas.area_atlases' }];
        scope.sceTabs.activaTab = 0;


        scope.$watch('esriConfig.urlLine', (newUrl) => {
          console.log('new url:' + newUrl);
          updateAtlasEsri0('line', newUrl);
        });

        scope.$watch('esriConfig.urlArea', (newUrl) => {
          updateAtlasEsri0('area', newUrl);
        });

        scope.$watch('esriConfig.urlAtlas', (newUrl) => {
          updateAtlasEsri0('atlas', newUrl);
        });



        // destroy the component, we call this from mapLeftMenu.js,
        // without the timeout it throws an error.
        scope.$on('destroy_atlas', function () {
          $timeout(() => scope.$destroy(), 0);
        });


        /**
         * Vérifie que les éléments saisis hormis le titre et la description
         * sont saisis afin de rendre accessible la commande de lancement
         * de l'impression de l'atlas.
         *
         * @returns VRAI s'il manque la saisie d'éléments, FAUX si on est prêt.
         */
        scope.notReadyForPrinting = () => {
          const config = scope.atlasEsri;
          return !(
            ( //-- Atlas surfacique + géométrie + PDF ou JPEG
              (config.type === 'area'
                && angular.isDefined(config.areaGeometry
                && config.area_export_type  ))
              //-- Atlas linéaire + géométrie + PDF ou JPEG
              || (config.type === 'line'
                && angular.isDefined(config.lineGeometry
                && config.line_export_type))
            )
            // -- Modèle d'atlas
            && config.selectedTemplate
          );
        };

      },
    };
  };

  atlas.$inject = ['AtlasFactory', '$filter', 'gclayers',
    'gaDomUtils', '$interval', '$q', '$rootScope', 'BaseMapFactory',
    '$location', '$translate', 'AtlasService', 'ngDialog', 'ConfigFactory',
    'gcStyleFactory', '$timeout', 'DataStoreFactory', 'pv2LayerServices',
    'gaJsUtils', 'PrintLegendFactory', 'extendedNgDialog', 'AtlasArcGIS108Service',
    'PortalsFactory'
  ];
  return atlas;
});