'use strict';

/**
 * For TEST MTA
 */
define(function() {
  var layersService = function(gclayers, mapJsUtils, $rootScope, $http) {

    let layerServices = {};

    // Objet contenant l'id du modèle de carte actif
    const resources = {activeModel: null};

    // constantes utilisées dans mapModelMgtWidget et gcMainController
    // KIS-2844: supprime "null" écrit après "MAP_MODELS_LAST_EDITED"
    const MAP_MODELS_USERS = 'MAP_MODELS_USERS';
    const MAP_MODEL_LAST_EDITED_CONFIGURATION_NAME
      = 'MAP_MODELS_LAST_EDITED'
      + ($rootScope.xgos.user && !$rootScope.xgos.isroot ? $rootScope.xgos.user.uid : '');

    /**
     * Alimentation d'une liste de groupes contenant des couches.
     * Chaque groupe est un storeName, donc un service de carte.
     * Chaque layer est une couche du dit service de carte.
     *
     * Une imlémentation UI est nécessaire, la layers open layers
     * génére des références circulaires. Pour gérer cirrectement
     * le drag and drog (User Interface), on ne doit pas
     * avoir de référence circulaire.
     *
     * @param layer
     * @param type
     * @param layerGroups
     * @param lg4ui
     */
    function addLayersToLayersForPriority(layer,type,layerGroups, lg4ui) {
      let groupObj, go4ui;
      const groupName = layer.get('fti').storeName;
      groupObj = layerServices[groupName];
      go4ui = layerServices[groupName+'_xx_ui'];
      if (groupObj == undefined) {
        groupObj = {
          group: groupName,
          layers: [],
          layers4ui: [],
          type: type,
          collapsed: true
        };
        go4ui = {
          group: groupName,
          layers4ui: [],
          type: type,
          collapsed: true
        };
        layerServices[groupName] = groupObj;
        layerServices[groupName+'_xx_ui'] = go4ui;
        layerGroups.push(groupObj);
        lg4ui.push(go4ui);
      }
      groupObj.layers.push(layer);
      go4ui.layers4ui.push(
        {
          name: layer.name,
          type: groupName,
          ftiUid: layer.fti.uid
        });
    }


    /**
     *   Au premier appel groups sera undefined.
     * Aux appels suivants, on devra recevoir en paramétre l'objet retourné
     * au premier appel pour l'actualiser. Cela permettra au layermanagerxidget
     * de s'actualiser tout seul/automatiquement.
     * @param {object} groups
     * @return {object} contient les groupes de couches openlayers et
     *                  les groupes de couches pour le géocatalogue
     */
    const  organizeLayers = (groups) => {
      layerServices = {};
      let layerGroups = [], lg4ui = [];
      if (groups != undefined) {
        layerGroups = groups.layerGroups;
        lg4ui = groups.layerGroups4ui;
        layerGroups.splice(0, layerGroups.length);
        lg4ui.splice(0, lg4ui.length);
      }
      let type;
      let layers = gclayers.getOperationalLayer();
      angular.forEach(layers,
        function (layer) {
          if (layer.gctype === 'esri') {
            type = 'ESRI';
          }
          else {
            type = 'WMS';
          }
          addLayersToLayersForPriority(layer, type, layerGroups, lg4ui);
        }
      );

      return {
        layerGroups: layerGroups,
        layerGroups4ui: lg4ui
      };

    };




    function getLayerGroupFromOlLayer(olLayer,groups) {
      let iGrp;
      const gr = groups.layerGroups;
      for (iGrp = 0; iGrp < gr.length; iGrp++) {
        if (gr[iGrp].group===olLayer.getProperties().groupOfLayers
          || gr[iGrp].mapOlLayer == olLayer)
          return gr[iGrp];
      }
      return null;
    }


    /**
     * Récupére le numéro de priorité de la couche dans la liste
     * de couches fournie. Le numéro de priorité est l'index de la couche
     * dans la dite liste.
     *
     * @param {*} layer : Couche dont on vzut connaître la priorité
     * @param {*} layerColl : Liste des couches triée par priorité de dessin
     * @returns Index de priorité de la couche
     */
    const getPriorityIndexInCollection = (layer, layerColl) => {
      let ftiUid;
      let ogcId;
      if (layer.getProperties) {
        ftiUid = layer.getProperties().fti.uid;
      }
      else {
        //-- Pour couche ESRI de gcmap.
        ogcId = layer.id;
      }
      let prio = 0;
      for (const aLayer of layerColl) {
        if (ogcId === aLayer.getProperties().fti.ogcId
                  || aLayer.getProperties().fti.uid === ftiUid) {
          return prio;
        }
        prio++;
      }
    };


    /**
     * Récupére le numéro de priorité d'une couche soit depuis la liste
     * des couches GeoServer, soit depuis la liste des couches ArcGIS.
     *
     * @param {*} layer Couche dont on veut connaître la priorité de dessin
     * @param {*} agLayerColl Liste des couches ArcGIS triée
     *                        par priorité de dessin
     * @param {*} gsLayerColl Liste des couches GeoServer triée
     *                        par priorité de dessin
     * @returns Indice de priorité de la couche
     */
    const getPriorityIndex = (layer) => {
      let priority;
      let priorityShift = 0;
      for (const group of gclayers.layersGroupedByStore.layerGroups) {
        priority = getPriorityIndexInCollection(layer, group.layers);
        if (priority===undefined) {
          priorityShift += group.layers.length;
        }
        else {
          return priority + priorityShift;
        }
      }
    };


    const isBackgroundLayer = (layerProperties) => {
      return layerProperties.type === 'WMSBACK'
        || (layerProperties.type === 'WMTS' && layerProperties.theme==='WebBackGround');
    };

    /**
     * Trie les couches à imprimer en fonction de leur priorité de dessin.
     * Les couches qui sont en tête de liste sont par dessus les suivantes.
     *
     * @param {*} layerArray Liste des couches à trier
     * @param {string} layerType 'esri' ou 'g2c'
     * @param {boolean} isPrinting true lorsque la méthode est exécutée
     *                          pour ordonner les couches à l'impression
     * @return tableau de couches trié en fonction de la priorité de dessin
     */
    const sortLayersOnPriority = (layerArray,layerType, isPrinting = false) => {
      const newList = [];
      const listWithPriority = [];
      const agLayerColl
                = gclayers.getOperationalLayerESRICollection().getArray();
      const gsLayerColl
                = gclayers.getOperationalLayerg2cCollection().getArray();
      for (const layer of layerArray) {
        let pp = {};
        if (layer.getProperties) {
          pp = layer.getProperties();
        }
        if (isBackgroundLayer(pp)) {
          listWithPriority.push( {priority: -1, layer: layer} );
        }
        else if ((layerType === 'esri' || pp.gctype === 'esri'
          || (pp.gctype === 'g2c' && pp.type!=='OSM')) && pp.name !== 'Mesure') {
          listWithPriority.push({
            priority: getPriorityIndex(layer, agLayerColl, gsLayerColl),
            layer: layer
          });
        }
        // KIS-3125:  les annotations créées sur la map lorsque visibles
        // doivent être imprimées sur les plans
        else if (layer.name === 'Edition') {
          listWithPriority.push({
            priority: -1,
            layer: layer
          });
        }
      }

      // Ajoute la couche "Mesure" et "Edition" (annotations) avec la plus haute priorité
      if (isPrinting) {
        setHighestPriorityToMeasureAndAnnotationLayers(layerArray, listWithPriority);
      }

      listWithPriority.sort((a, b) => {
        return a.priority - b.priority;
      });
      for (const layerWithPrio of listWithPriority) {
        newList.push(layerWithPrio.layer);
      }
      return newList;
    };

    /**
     * Active la visibilité de la couche technique de sélection
     * KIS-2853
     */
    const enableSelectLayerVisible = (map) => {
      const selectLayer = mapJsUtils.getG2cLayerByProperty(map, 'name', 'Selection');
      if (selectLayer && !selectLayer.get('visible')) {
        selectLayer.set('visible', true);
      }
    };

    /**
     * Si elle est présente dans le tableau de couche fourni en paramètre,
     * la couche mesure est insérée dans un autre tableau de couche avec
     * la plus haute valeur de priorité possible
     * afin que les cotations se trouvent au-dessus des autres couches.
     * KIS-3232: traite de la même manière les couches "Edition" des annotations
     * @param initialLayersArray tableau des couches à imprimer
     * @param layersArrayPriority tableau des couches à imprimer avec gestion de la priorité
     */
    const setHighestPriorityToMeasureAndAnnotationLayers
      = (initialLayersArray, layersArrayPriority) => {
        const maxPriorityObject
          = layersArrayPriority.reduce(
            (prev, current) => (prev.priority > current.priority) ? prev : current);

        const measureLayersNames = ['Mesure', 'measure-tooltip',
          'measure-tooltip-pointer', 'Edition'];

        for (let i = 0; i < measureLayersNames.length; i++) {
          const measureLayerName = measureLayersNames[i];

          if (measureLayerName === 'Edition') {
          // couches des annotations (cf. pv2annotations.addAnnotationsAsLayer)
          // une couche "Edition" pour chaque annotation
            const annotLayers = initialLayersArray.filter(layer => layer.name === measureLayerName);
            for (let j = 0; j < annotLayers.length; j++) {
              const annotLayer = annotLayers[j];
              if (annotLayer) {
                layersArrayPriority.push({
                  priority: maxPriorityObject.priority + i + j + 1,
                  layer: annotLayer
                });
              }
            }
          } else {
          // couches des cotations (cf. MeasureFactory.addMeasureToolLayers)
            const measureLayer = initialLayersArray.find(
              layer => layer.get('name') === measureLayerName);
            if (measureLayer) {
              layersArrayPriority.push({
                priority: maxPriorityObject.priority + i + 1,
                layer: measureLayer
              });
            }
          }
        }
      };

    /**
     * Applique à la carte le mapModel fourni en paramètre.
     * Méthode rapatriée depuis <code>mapModelMgtWidget</code>
     * @param {object} mapmodel paramètre KIS de type mapModel
     */
    const setMapModel = (mapmodel) => {
      $rootScope.$broadcast('gcOperationalLayerEmptyToUpdate');
      gclayers.clearLayersFromGroupLayer();
      gclayers.clearOperationalLayerg2c(true);
      gclayers.clearOperationalLayerESRI(true);

      // sauvegarde le modèle de carte actif
      console.log(`sauvegarde model actif ${mapmodel.data.name} : `, mapmodel.data);
      resources.activeModel = mapmodel.data.id;

      $rootScope.$broadcast('gcUpdateMainConfig', mapmodel.data, true);
      $rootScope.$broadcast('updateLayersManagers');
    };

    /**
     * Exécute la requête "legend" de l'api ArcGIS.
     * Sert à afficher l'image de la représentation d'une couche dans le géocatalogue
     * @param url chemin pour effectuer la requête
     * @return {Promise} objet décrivant une légende contenant les images de chaque item
     */
    const getArcGisLegendInfo = (url) => {
      return $http.get(url).catch(e=>console.error(e));
    };

    return {
      MAP_MODELS_USERS: MAP_MODELS_USERS,
      MAP_MODEL_LAST_EDITED_CONFIGURATION_NAME: MAP_MODEL_LAST_EDITED_CONFIGURATION_NAME,
      resources: resources,
      organizeLayers: organizeLayers,
      getLayerGroupFromOlLayer: getLayerGroupFromOlLayer,
      sortLayersOnPriority: sortLayersOnPriority,
      enableSelectLayerVisible: enableSelectLayerVisible,
      setMapModel: setMapModel,
      getArcGisLegendInfo: getArcGisLegendInfo,
      isBackgroundLayer: isBackgroundLayer
    };
  };

  layersService.$inject = ['gclayers', 'mapJsUtils', '$rootScope', '$http'];

  return layersService;
});
